package travelersclient

import (
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/travel/app/backend/internal/common"
)

type Gender string

type field string

const (
	fieldDocuments  field = "documents"
	fieldBonusCards field = "bonus-cards"
)
const fieldsParamName = "fields"

type DocumentTypeInfo struct {
	Unused        []string          `json:"unused"`
	Required      []string          `json:"required"`
	ReValidations map[string]string `json:"re_validations"`
}

type DocumentTypes map[string]DocumentTypeInfo

type Traveler struct {
	Email           string     `json:"email"`
	Phone           string     `json:"phone"`
	PhoneAdditional string     `json:"phone_additional"`
	Agree           bool       `json:"agree"`
	CreatedAt       *Timestamp `json:"created_at"`
	UpdatedAt       *Timestamp `json:"updated_at"`
}

type EditableTraveler struct {
	Email           string `json:"email,omitempty"`
	Phone           string `json:"phone,omitempty"`
	PhoneAdditional string `json:"phone_additional,omitempty"`
	Agree           bool   `json:"agree"`
}

var passengerFieldNames = []string{
	"title",
	"gender",
	"birth_date",
	"itn",
	"phone",
	"phone_additional",
	"email",
	"train_notifications_enabled",
}

type rawPassenger map[string]interface{}

type Passenger struct {
	ID         string
	Fields     map[string]interface{}
	CreatedAt  *Timestamp
	UpdatedAt  *Timestamp
	Documents  []Document
	BonusCards []BonusCard
}

type CreateOrUpdatePassengerRequest struct {
	Fields map[string]interface{}
}

var documentFieldNames = []string{
	"title",
	"number",
	"first_name",
	"middle_name",
	"last_name",
	"first_name_en",
	"middle_name_en",
	"last_name_en",
	"issue_date",
	"expiration_date",
	"citizenship",
}

type rawDocument map[string]interface{}

type CreateOrUpdateDocumentRequest struct {
	Type   string
	Fields map[string]interface{}
}

type Document struct {
	ID          string
	PassengerID string
	Type        string
	CreatedAt   *Timestamp
	UpdatedAt   *Timestamp
	Fields      map[string]interface{}
}

var bonusCardFieldNames = []string{"title", "number"}

type rawBonusCard map[string]interface{}

type BonusCard struct {
	ID          string
	PassengerID string
	Type        string
	Fields      map[string]interface{}
	CreatedAt   *Timestamp
	UpdatedAt   *Timestamp
}

type EditableBonusCard struct {
	Type   string
	Fields map[string]interface{}
}

type Timestamp struct {
	time.Time
}

type Date struct {
	time.Time
}

const travelersTimestampPattern = "2006-01-02 15:04:05"

var moscowLocation *time.Location

func (t *Timestamp) UnmarshalJSON(b []byte) (err error) {
	s := strings.Trim(string(b), "\"")
	if s == "null" {
		t.Time = time.Time{}
		return
	}
	tt, err := time.ParseInLocation(travelersTimestampPattern, s, moscowLocation)
	if err != nil {
		return err
	}
	t.Time = tt
	return nil
}

func (d *Date) UnmarshalJSON(b []byte) (err error) {
	s := strings.Trim(string(b), "\"")
	if s == "null" {
		d.Time = time.Time{}
		return
	}
	t, err := common.ParseDate(s)
	if err != nil {
		// travelers api may get invalid time, i.e. "1-01-01. We should not return 500 in this case, so we just return empty time
		d.Time = time.Time{}
		return nil
	}
	d.Time = t
	return nil
}

func (d *Date) MarshalJSON() ([]byte, error) {
	if d == nil {
		return nil, nil
	}
	return []byte(fmt.Sprintf("%q", common.FormatDate(d.Time))), nil
}

func init() {
	moscowLocation, _ = time.LoadLocation("Europe/Moscow")
}
