package handler

import (
	"context"
	"fmt"
	"strconv"

	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	personalizationAPI "a.yandex-team.ru/travel/app/backend/api/personalization/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/points"
	personalsearch "a.yandex-team.ru/travel/avia/personalization/api/personal_search/v2"
)

const (
	searchShortcutsTitle       = "search shortcuts"
	searchAviaShortcutsTitle   = "search shortcuts avia"
	searchHotelsShortcutsTitle = "search shortcuts hotels"
)

type GRPCPersonalizationHandler struct {
	logger                    log.Logger
	aviaPersonalizationClient personalsearch.PersonalizationServiceV2Client
	pointsParser              points.IParser
}

func NewGRPCPersonalizationHandler(
	logger log.Logger,
	aviaPersonalizationClient personalsearch.PersonalizationServiceV2Client,
	pointsParser points.IParser,
) *GRPCPersonalizationHandler {
	h := GRPCPersonalizationHandler{
		logger:                    logger,
		aviaPersonalizationClient: aviaPersonalizationClient,
		pointsParser:              pointsParser,
	}
	return &h
}

func (h *GRPCPersonalizationHandler) SearchShortcuts(ctx context.Context, req *personalizationAPI.SearchShortcutsReq) (*personalizationAPI.SearchShortcutsRsp, error) {
	enabledShortcuts := make(map[personalizationAPI.ShortcutsType]bool)
	enableAll := len(req.EnabledShortcuts) == 0
	for _, sType := range req.EnabledShortcuts {
		enabledShortcuts[sType] = true
	}
	auth := common.GetAuth(ctx)
	if auth == nil || auth.User == nil {
		return &personalizationAPI.SearchShortcutsRsp{}, nil
	}
	limit := req.Limit
	if limit <= 0 || limit > 100 {
		limit = 100
	}
	passportID := strconv.FormatUint(auth.User.ID, 10)

	aviaChan := make(chan *personalizationAPI.AviaSearchShortcut)
	if _, ok := enabledShortcuts[personalizationAPI.ShortcutsType_SHORTCUT_TYPE_AVIA]; ok || enableAll {
		go h.getAviaShortcuts(ctx, passportID, limit, aviaChan)
	} else {
		close(aviaChan)
	}
	hotelsChan := make(chan *personalizationAPI.HotelsSearchShortcut)
	if _, ok := enabledShortcuts[personalizationAPI.ShortcutsType_SHORTCUT_TYPE_HOTELS]; ok || enableAll {
		go h.getHotelsShortcuts(ctx, passportID, limit, hotelsChan)
	} else {
		close(hotelsChan)
	}

	var aviaShortcuts []*personalizationAPI.AviaSearchShortcut
	for aviaShortcut := range aviaChan {
		aviaShortcuts = append(aviaShortcuts, aviaShortcut)
	}
	var hotelsShortcuts []*personalizationAPI.HotelsSearchShortcut
	for hotelsShortcut := range hotelsChan {
		hotelsShortcuts = append(hotelsShortcuts, hotelsShortcut)
	}
	return &personalizationAPI.SearchShortcutsRsp{
		AviaShortcuts:   aviaShortcuts,
		HotelsShortcuts: hotelsShortcuts,
	}, nil
}

func (h *GRPCPersonalizationHandler) getAviaShortcuts(ctx context.Context, passportID string, limit uint32,
	ch chan<- *personalizationAPI.AviaSearchShortcut) {
	rsp, err := h.aviaPersonalizationClient.GetAviaHistory(ctx, &personalsearch.TGetAviaHistoryRequestV2{
		PassportId: passportID,
		Limit:      limit,
	})
	if err != nil {
		msg := fmt.Sprintf("%s aviaPersonalizationClient.GetAviaHistory error", searchAviaShortcutsTitle)
		ctxlog.Error(ctx, h.logger, msg, log.Error(err))
	}
	for _, entry := range rsp.GetEntries() {
		aviaItem := entry.GetAvia()
		if aviaItem == nil {
			continue
		}
		dateForward, err := common.DateStringToProto(aviaItem.When)
		if err != nil {
			msg := fmt.Sprintf("%s error parse date \"%s\", passportID=%s", searchAviaShortcutsTitle, aviaItem.When, passportID)
			ctxlog.Error(ctx, h.logger, msg, log.Error(err))
			continue
		}
		dateBackward, err := common.DateStringToProto(aviaItem.ReturnDate)
		if err != nil {
			msg := fmt.Sprintf("%s error parse date \"%s\", passportID=%s", searchAviaShortcutsTitle, aviaItem.ReturnDate, passportID)
			ctxlog.Error(ctx, h.logger, msg, log.Error(err))
			continue
		}
		locale := common.GetLocale(ctx)
		pointFrom, err := h.pointsParser.ParseByPointKey(aviaItem.PointFrom.PointCode)
		if err != nil {
			msg := fmt.Sprintf("%s error parse point \"%s\", passportID=%s", searchAviaShortcutsTitle, aviaItem.PointFrom.PointCode, passportID)
			ctxlog.Error(ctx, h.logger, msg, log.Error(err))
			continue
		}
		pointFromTitle := pointFrom.Title(locale.Language)
		pointTo, err := h.pointsParser.ParseByPointKey(aviaItem.PointTo.PointCode)
		if err != nil {
			msg := fmt.Sprintf("%s error parse point \"%s\", passportID=%s", searchAviaShortcutsTitle, aviaItem.PointTo.PointCode, passportID)
			ctxlog.Error(ctx, h.logger, msg, log.Error(err))
			continue
		}
		serviceClass := classStringToProto(aviaItem.AviaClass)
		if serviceClass == aviaAPI.ServiceClass_SERVICE_CLASS_UNKNOWN {
			msg := fmt.Sprintf("%s error parse service class \"%s\", passportID=%s", searchAviaShortcutsTitle, aviaItem.AviaClass, passportID)
			ctxlog.Error(ctx, h.logger, msg, log.Error(err))
			continue
		}
		pointToTitle := pointTo.Title(locale.Language)
		ch <- &personalizationAPI.AviaSearchShortcut{
			SearchParams: &aviaAPI.InitSearchReq{
				ServiceClass: serviceClass,
				Passengers: &aviaAPI.Passengers{
					Adults:   uint32(aviaItem.Travelers.Adults),
					Children: uint32(aviaItem.Travelers.Children),
					Infants:  uint32(aviaItem.Travelers.Infants),
				},
				PointKeyFrom: aviaItem.PointFrom.PointCode,
				PointKeyTo:   aviaItem.PointTo.PointCode,
				DateForward:  dateForward,
				DateBackward: dateBackward,
			},
			TitleFrom: pointFromTitle,
			TitleTo:   pointToTitle,
		}
	}
	close(ch)
}

func classStringToProto(classString string) aviaAPI.ServiceClass {
	switch classString {
	case "business":
		return aviaAPI.ServiceClass_SERVICE_CLASS_BUSINESS
	case "economy":
		return aviaAPI.ServiceClass_SERVICE_CLASS_ECONOMY
	}
	return aviaAPI.ServiceClass_SERVICE_CLASS_UNKNOWN
}

func (h *GRPCPersonalizationHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		personalizationAPI.RegisterPersonalizationAPIServer(server, h)
	}
}

func (h *GRPCPersonalizationHandler) getHotelsShortcuts(ctx context.Context, passportID string, limit uint32,
	ch chan<- *personalizationAPI.HotelsSearchShortcut) {
	rsp, err := h.aviaPersonalizationClient.GetHotelsSuggest(ctx, &personalsearch.TGetHotelsSuggestRequestV2{
		PassportId:    passportID,
		OrdersLimit:   limit,
		SearchesLimit: limit,
	})
	if err != nil {
		msg := fmt.Sprintf("%s aviaPersonalizationClient.GetHotelsSuggest error", searchHotelsShortcutsTitle)
		ctxlog.Error(ctx, h.logger, msg, log.Error(err))
	}
	var count uint32 = 0
	for _, entry := range rsp.GetEntries() {
		if count >= limit {
			break
		}
		hotelItem := entry.GetHotel()
		if hotelItem == nil || hotelItem.PointTo == nil {
			continue
		}
		checkInDate, err := common.DateStringToProto(hotelItem.CheckInDate)
		if err != nil {
			msg := fmt.Sprintf("%s error parse date \"%s\", passportID=%s", searchHotelsShortcutsTitle, hotelItem.CheckInDate, passportID)
			ctxlog.Error(ctx, h.logger, msg, log.Error(err))
			continue
		}
		checkOutDate, err := common.DateStringToProto(hotelItem.CheckOutDate)
		if err != nil {
			msg := fmt.Sprintf("%s error parse date \"%s\", passportID=%s", searchHotelsShortcutsTitle, hotelItem.CheckOutDate, passportID)
			ctxlog.Error(ctx, h.logger, msg, log.Error(err))
			continue
		}
		locale := common.GetLocale(ctx)
		point, err := h.pointsParser.ParseByPointKey(hotelItem.PointTo.PointCode)
		if err != nil {
			msg := fmt.Sprintf("%s error parse point \"%s\", passportID=%s", searchHotelsShortcutsTitle, hotelItem.PointTo.PointCode, passportID)
			ctxlog.Error(ctx, h.logger, msg, log.Error(err))
			continue
		}
		var childrenAges []uint32
		for _, age := range hotelItem.Travelers.ChildrenAges {
			childrenAges = append(childrenAges, uint32(age))
		}
		pointTitle := point.Title(locale.Language)
		ch <- &personalizationAPI.HotelsSearchShortcut{
			CheckinDate:  checkInDate,
			CheckoutDate: checkOutDate,
			Adults:       uint32(hotelItem.Travelers.Adults),
			ChildrenAges: childrenAges,
			GeoId:        int32(hotelItem.PointTo.GeoId),
			PointKey:     hotelItem.PointTo.PointCode,
			PointTitle:   pointTitle,
		}
		count++
	}
	close(ch)
}
