package points

import (
	"strconv"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/references"
)

var UnexpectedFormatErr = xerrors.New("unexpected key format")
var NotExistsErr = xerrors.New("geo object does not exist")

type IParser interface {
	ParseByPointKey(key string) (point Point, err error)
}

type Parser struct {
	registry *references.Registry
}

func NewParser(registry *references.Registry) *Parser {
	return &Parser{registry: registry}
}

func (p *Parser) ParseByPointKey(key string) (point Point, err error) {
	defer func() {
		if err != nil {
			err = xerrors.Errorf("ParseByPointKey(%s): %w", key, err)
		}
	}()

	if len(key) == 1 {
		return nil, UnexpectedFormatErr
	}

	prefix := key[0:1]
	pointID, err := strconv.Atoi(key[1:])
	if err != nil {
		return nil, UnexpectedFormatErr
	}

	switch prefix {
	case "c":
		settlementRepo := p.registry.Settlements
		if settlement, found := settlementRepo.Get(pointID); found {
			return NewSettlement(settlement), nil
		}
	case "s":
		stationRepo := p.registry.Stations
		if station, found := stationRepo.Get(pointID); found {
			return NewStation(station), nil
		}
	default:
		return nil, UnexpectedFormatErr
	}

	return nil, NotExistsErr
}
