package points

import (
	"strconv"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type PointType int

const (
	SettlementType PointType = iota
	StationType
)

type Point interface {
	Proto() proto.Message
	Type() PointType
	ID() int32
	Slug() string
	PointKey() string
	Title(language string) string
	SettlementID() *int32
}

type Station struct {
	value *rasp.TStation
}

func NewStation(value *rasp.TStation) Point {
	if value == nil {
		return nil
	}
	return &Station{value}
}

func (p *Station) Proto() proto.Message {
	return p.value
}

func (p *Station) Type() PointType {
	return StationType
}

func (p *Station) ID() int32 {
	return p.value.Id
}

func (p *Station) SettlementID() *int32 {
	if p.value.SettlementId > 0 {
		return &p.value.SettlementId
	}
	return nil
}

func (p *Station) Slug() string {
	return p.value.Slug
}

func (p *Station) PointKey() string {
	return "s" + strconv.Itoa(int(p.ID()))
}

func (p *Station) Title(language string) string {
	switch language {
	case "ru":
		if p.value.TitleRuNominativeCase != "" {
			return p.value.TitleRuNominativeCase
		}
	case "uk":
		if p.value.TitleUkNominativeCase != "" {
			return p.value.TitleUkNominativeCase
		}
	}
	return p.value.TitleDefault
}

type Settlement struct {
	value *rasp.TSettlement
}

func NewSettlement(value *rasp.TSettlement) *Settlement {
	if value == nil {
		return nil
	}
	return &Settlement{value}
}

func (p *Settlement) Proto() proto.Message {
	return p.value
}

func (p *Settlement) Type() PointType {
	return SettlementType
}

func (p *Settlement) ID() int32 {
	return p.value.Id
}

func (p *Settlement) SettlementID() *int32 {
	return &p.value.Id
}

func (p *Settlement) Slug() string {
	return p.value.Slug
}

func (p *Settlement) PointKey() string {
	return "c" + strconv.Itoa(int(p.ID()))
}

func (p *Settlement) Title(language string) string {
	switch language {
	case "ru":
		if p.value.Title.Ru.Nominative != "" {
			return p.value.Title.Ru.Nominative
		}
	case "uk":
		if p.value.Title.Uk.Nominative != "" {
			return p.value.Title.Uk.Nominative
		}
	case "en":
		if p.value.Title.En.Nominative != "" {
			return p.value.Title.En.Nominative
		}
	}

	return p.value.TitleDefault
}
