package references

import (
	"strconv"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/dicts/repository"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type CountryRepository struct {
	baseRepository *repository.CountryRepository
	countryByGeoID map[int]*rasp.TCountry
}

func NewCountryRepository() *CountryRepository {
	return &CountryRepository{
		baseRepository: repository.NewCountryRepository(),
		countryByGeoID: make(map[int]*rasp.TCountry),
	}
}

func (cr *CountryRepository) Write(b []byte) (int, error) {
	country := &rasp.TCountry{}
	if err := proto.Unmarshal(b, country); err != nil {
		return 0, xerrors.Errorf("CountryRepository:Write %w", err)
	}
	cr.Add(country)
	return len(b), nil
}

func (cr *CountryRepository) Add(country *rasp.TCountry) {
	cr.baseRepository.Add(country)
	if _, ok := cr.countryByGeoID[int(country.GetGeoId())]; !ok {
		cr.countryByGeoID[int(country.GetGeoId())] = country
	}
}

func (cr *CountryRepository) Get(id int) (*rasp.TCountry, bool) {
	country, ok := cr.baseRepository.Get(int32(id))
	return country, ok
}

func (cr *CountryRepository) GetByGeoID(id int) (*rasp.TCountry, bool) {
	country, found := cr.countryByGeoID[id]
	return country, found
}

func (cr *CountryRepository) GetPointKeyByGeoID(id int) (string, bool) {
	result, found := cr.GetByGeoID(id)
	if result == nil || !found {
		return "", false
	}
	return "l" + strconv.Itoa(int(result.Id)), found
}

func (cr *CountryRepository) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewCountryRepository()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*cr = *newState
	return nil
}
