package references

import (
	"strconv"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/dicts/repository"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type RegionRepository struct {
	baseRepository *repository.RegionRepository
	regionByGeoID  map[int]*rasp.TRegion
}

func NewRegionRepository() *RegionRepository {
	return &RegionRepository{
		baseRepository: repository.NewRegionRepository(),
		regionByGeoID:  make(map[int]*rasp.TRegion),
	}
}

func (cr *RegionRepository) Write(b []byte) (int, error) {
	region := &rasp.TRegion{}
	if err := proto.Unmarshal(b, region); err != nil {
		return 0, xerrors.Errorf("RegionRepository:Write: %w", err)
	}
	cr.Add(region)
	return len(b), nil
}

func (cr *RegionRepository) Add(region *rasp.TRegion) {
	cr.baseRepository.Add(region)

	if _, ok := cr.regionByGeoID[int(region.GetGeoId())]; !ok {
		cr.regionByGeoID[int(region.GetGeoId())] = region
	}
}

func (cr *RegionRepository) Get(id int) (*rasp.TRegion, bool) {
	region, ok := cr.baseRepository.Get(id)
	return region, ok
}

func (cr *RegionRepository) GetPointKeyByGeoID(id int) (string, bool) {
	result, found := cr.regionByGeoID[id]
	if result == nil || !found {
		return "", false
	}
	return "r" + strconv.Itoa(int(result.Id)), found
}

func (cr *RegionRepository) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewRegionRepository()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*cr = *newState
	return nil
}
