package references

import (
	"io"
	"path/filepath"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/dicts/base"
)

type Config struct {
	ResourcesPath string `config:"DICTS_RESOURCES_PATH" yaml:"resources_path"`
}

var DefaultConfig = Config{
	ResourcesPath: "/data/shared-dicts",
}

type Registry struct {
	config       Config
	logger       log.Logger
	writerByPath map[string]UpdatableWriter
	Stations     *StationsRepository
	Settlements  *SettlementsRepository
	Regions      *RegionRepository
	Countries    *CountryRepository
}

type PointKeyByGeoIDGetter interface {
	GetPointKeyByGeoID(id int) (string, bool)
}

type RepositoryUpdater interface {
	Populate(writer io.Writer) error
}

type UpdatableWriter interface {
	io.Writer
	UpdateFromSource(RepositoryUpdater) error
}

func NewRegistry(c Config, logger log.Logger) (*Registry, error) {
	const funcName = "internal.references:NewRegistry"
	r := &Registry{
		config:      c,
		logger:      logger,
		Stations:    NewStationsRepository(),
		Settlements: NewSettlementsRepository(),
		Regions:     NewRegionRepository(),
		Countries:   NewCountryRepository(),
	}

	resourcePath, err := getResourcePath(c)
	if err != nil {
		return nil, xerrors.Errorf("%s: %w", funcName, err)
	}

	r.updateWriterByPath(resourcePath)
	err = r.populateMany(r.writerByPath)
	if err != nil {
		return nil, xerrors.Errorf("%s: %w", funcName, err)
	}
	return r, nil
}

func (r *Registry) ID() string {
	return "references"
}

func (r *Registry) Reload() error {
	const funcName = "internal.references.registry:Reload"
	resourcePath, err := getResourcePath(r.config)
	if err != nil {
		return xerrors.Errorf("%s: %w", funcName, err)
	}
	r.updateWriterByPath(resourcePath)
	err = r.populateMany(r.writerByPath)
	if err != nil {
		return xerrors.Errorf("%s: %w", funcName, err)
	}
	return err
}

func (r *Registry) updateWriterByPath(resourcePath string) {
	r.writerByPath = map[string]UpdatableWriter{
		resourcePath + "/station.data":    r.Stations,
		resourcePath + "/settlement.data": r.Settlements,
		resourcePath + "/region.data":     r.Regions,
		resourcePath + "/country.data":    r.Countries,
	}
}

func (r *Registry) populateMany(path2writer map[string]UpdatableWriter) error {
	const funcName = "internal.references.registry:populateMany"
	for path, writer := range path2writer {
		bytesIterator, err := base.BuildIteratorFromFile(path)
		if err != nil {
			return xerrors.Errorf("%s: %w", funcName, err)
		}
		err = bytesIterator.Populate(writer)
		if err != nil {
			return xerrors.Errorf("%s: %w", funcName, err)
		}
		r.logger.Infof("%s: path %s updated", funcName, path)
	}
	return nil
}

func (r *Registry) GetPointKeyByGeoID(id int) (string, bool) {
	getters := []PointKeyByGeoIDGetter{r.Settlements, r.Regions, r.Countries}
	for _, g := range getters {
		res, exists := g.GetPointKeyByGeoID(id)
		if exists {
			return res, true
		}
	}
	return "", false
}

func getResourcePath(config Config) (string, error) {
	target, err := filepath.EvalSymlinks(config.ResourcesPath)
	if err != nil {
		return "", err
	}
	return filepath.Dir(target), nil
}
