package handler

import (
	"context"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	srvCfg "a.yandex-team.ru/travel/app/backend/api/serverconfig/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
	"a.yandex-team.ru/travel/app/backend/internal/serverconfig"
)

type GRPCServerConfigHandler struct {
	logger  log.Logger
	service *serverconfig.Service
	l10n    *l10n.Service
}

func NewGRPCServerConfigHandler(l log.Logger, service *serverconfig.Service, l10n *l10n.Service) *GRPCServerConfigHandler {
	return &GRPCServerConfigHandler{
		logger:  l,
		service: service,
		l10n:    l10n,
	}
}

func (h *GRPCServerConfigHandler) GetServerConfig(ctx context.Context, req *srvCfg.ServerConfigReq) (*srvCfg.ServerConfigRsp, error) {
	userAgent := common.GetUserAgent(ctx)
	if userAgent.KnownOS == common.OSTypeUnknown {
		ctxlog.Warn(ctx, h.logger, "unsupported OS type")
		return nil, status.Error(codes.InvalidArgument, "unsupported OS type")
	}
	config, tag, err := h.service.GetServerConfigByPlatform(ctx, userAgent.KnownOS, userAgent.AppVersion, userAgent.AppVersionCode, req.Tag, req.Debug)
	if err != nil {
		ctxlog.Warn(ctx, h.logger, "not found server config", log.Error(err))
		return nil, status.Error(codes.NotFound, err.Error())
	}
	return &srvCfg.ServerConfigRsp{
		Tag:    tag,
		Debug:  req.Debug,
		Config: config,
	}, nil
}

func (h *GRPCServerConfigHandler) GetDynamicLocalization(ctx context.Context, req *srvCfg.DynamicLocalizationReq) (*srvCfg.DynamicLocalizationRsp, error) {
	lang := common.GetLanguage(ctx)

	ks, err := h.l10n.Get(req.Keyset, lang)
	if err != nil {
		return nil, status.Error(codes.NotFound, err.Error())
	}
	var localizations map[string]string
	if ks.Tag != req.Tag {
		localizations = ks.Keys
	}
	return &srvCfg.DynamicLocalizationRsp{
		Keyset:        ks.Name,
		Language:      ks.Language,
		Tag:           ks.Tag,
		Localizations: localizations,
	}, nil
}

func (h *GRPCServerConfigHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		srvCfg.RegisterServerConfigAPIServer(server, h)
	}
}
