package serverconfig

import (
	"bytes"
	"context"
	"encoding/json"
	"io/ioutil"
	"sync"

	"github.com/golang/protobuf/jsonpb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	srvCfg "a.yandex-team.ru/travel/app/backend/api/serverconfig/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/exp3matcher"
	"a.yandex-team.ru/travel/app/backend/pkg/hashutils"
)

type Config struct {
	Path  string `config:"server-config-path,required"`
	Debug bool   `config:"server-config-debug"`
}

type Service struct {
	cfg           Config
	environment   common.EnvType
	logger        log.Logger
	serverConfigs *compositeParsedServerConfig
	mutex         sync.RWMutex
	exp3client    *exp3matcher.HTTPClient
}

type rawServerConfig struct {
	Common  map[string]interface{} `json:"common"`
	IOS     map[string]interface{} `json:"ios"`
	Android map[string]interface{} `json:"android"`
}

type taggedRawServerConfig struct {
	Config rawServerConfig `json:"config"`
}

type compositeRawServerConfig struct {
	Primary taggedRawServerConfig `json:"primary"`
	// Debug используется, когда в тестовом окружении запрашиваем с флагом debug.
	Debug taggedRawServerConfig `json:"debug"`
	// Testing используется, когда для тестового окружения нужно какое-то переопределение.
	// Например, чтобы в тестинге отдать URL тестового портала.
	Testing taggedRawServerConfig `json:"testing"`
}

type parsedServerConfig struct {
	Ios     *srvCfg.ServerConfig
	Android *srvCfg.ServerConfig
}

type compositeParsedServerConfig struct {
	Primary *parsedServerConfig
	Debug   *parsedServerConfig
}

func (s *Service) logAndWrapError(err error) error {
	wrapped := xerrors.Errorf("unable to load server-config: %w", err)
	s.logger.Error(wrapped.Error(), log.Error(err))
	return wrapped
}

func (s *Service) ID() string {
	return "server-config"
}

func (s *Service) Reload() error {
	var err error
	s.logger.Info("Reloading server config")
	configBytes, err := ioutil.ReadFile(s.cfg.Path)
	if err != nil {
		return xerrors.Errorf("unable to open server config file: %w", err)
	}
	var rawConfig compositeRawServerConfig
	if err = json.Unmarshal(configBytes, &rawConfig); err != nil {
		return xerrors.Errorf("unable to load server config: %w", err)
	}

	if primIOS, primAndroid, debugIOS, debugAndroid, err := buildMaps(rawConfig, s.cfg.Debug, s.environment); err != nil {
		return xerrors.Errorf("unable to build maps out of raw config: %w", err)
	} else {
		if result, err := parseConfigs(primIOS, primAndroid, debugIOS, debugAndroid, s.cfg.Debug); err != nil {
			return err
		} else {
			s.mutex.Lock()
			defer s.mutex.Unlock()
			s.serverConfigs = result
			s.logger.Info("server config reloaded")
			return nil
		}
	}
}

func parseConfig(raw map[string]interface{}) (*srvCfg.ServerConfig, error) {
	mergedBytes, err := json.Marshal(raw)
	if err != nil {
		return nil, xerrors.Errorf("unable to marshal merged structure: %w", err)
	}
	var res srvCfg.ServerConfig
	if err := jsonpb.Unmarshal(bytes.NewReader(mergedBytes), &res); err != nil {
		return nil, xerrors.Errorf("unable to parse merged structure: %w", err)
	}
	return &res, nil
}

func NewService(cfg Config, env common.EnvType, logger log.Logger, exp3client *exp3matcher.HTTPClient) (*Service, error) {
	s := Service{cfg: cfg, environment: env, logger: logger, exp3client: exp3client}
	if err := s.Reload(); err != nil {
		return nil, err
	}
	return &s, nil
}

func (s *Service) GetServerConfigByPlatform(ctx context.Context, osType common.OSType, appVersion string, appVersionCode uint64, tag string, debug bool) (*srvCfg.ServerConfig, string, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var source *parsedServerConfig
	if debug {
		if source = s.serverConfigs.Debug; source == nil {
			return nil, "", xerrors.Errorf("debug server config is not supported at this environment")
		}
	} else {
		source = s.serverConfigs.Primary
	}
	var res *srvCfg.ServerConfig
	switch osType {
	case common.OSTypeIOS:
		res = source.Ios
	case common.OSTypeAndroid:
		res = source.Android
	}
	if res == nil {
		return nil, "", xerrors.Errorf("no server config for platform %s", osType.Name())
	}

	updInfoChan := make(chan *srvCfg.UpdateInfo, 1)
	retryPolicyChan := make(chan *srvCfg.RetryPolicyCfg, 1)
	userProfileChan := make(chan *srvCfg.UserProfileConfig, 1)
	go func() {
		updInfoChan <- s.exp3client.GetUpdateInfoConfig(ctx)
		close(updInfoChan)
	}()
	go func() {
		retryPolicyChan <- s.exp3client.GetRetryPolicyConfig(ctx)
		close(retryPolicyChan)
	}()
	go func() {
		userProfileChan <- s.exp3client.GetUserProfileConfig(ctx)
		close(userProfileChan)
	}()
	res.UpdateInfo = <-updInfoChan
	if debug && res.UpdateInfo != nil {
		res.UpdateInfo.UpdateType = srvCfg.UpdateType_UPDATE_TYPE_NOT
	}
	retryPolicy := <-retryPolicyChan
	if retryPolicy != nil {
		res.RetryPolicyCfg = retryPolicy
	}
	userProfile := <-userProfileChan
	if userProfile != nil {
		res.UserProfile = userProfile
	}
	hash, err := hashutils.GetSimpleHashFromStruct(res)
	if err != nil {
		return nil, "", xerrors.Errorf("getting hash error: %w", err)
	}
	if hash == tag {
		return nil, hash, nil
	}
	return res, hash, nil
}

func parseConfigs(primIOS, primAndroid, debugIOS, debugAndroid map[string]interface{}, hasDebug bool) (*compositeParsedServerConfig, error) {
	var prodIOSParsed, prodAndroidParsed, debugIOSParsed, debugAndroidParsed *srvCfg.ServerConfig
	var err error
	if prodIOSParsed, err = parseConfig(primIOS); err != nil {
		return nil, xerrors.Errorf("unable to parse primary ios config: %w", err)
	}
	if prodAndroidParsed, err = parseConfig(primAndroid); err != nil {
		return nil, xerrors.Errorf("unable to parse primary android config: %w", err)
	}
	if hasDebug {
		if debugIOSParsed, err = parseConfig(debugIOS); err != nil {
			return nil, xerrors.Errorf("unable to parse debug ios config: %w", err)
		}
		if debugAndroidParsed, err = parseConfig(debugAndroid); err != nil {
			return nil, xerrors.Errorf("unable to parse debug android config: %w", err)
		}
	}
	result := compositeParsedServerConfig{
		Primary: &parsedServerConfig{
			Ios:     prodIOSParsed,
			Android: prodAndroidParsed,
		},
	}
	if hasDebug {
		result.Debug = &parsedServerConfig{
			Ios:     debugIOSParsed,
			Android: debugAndroidParsed,
		}
	}
	return &result, nil
}

func buildMaps(config compositeRawServerConfig, hasDebug bool, env common.EnvType) (primIOS, primAndroid, debugIOS, debugAndroid map[string]interface{}, err error) {
	if env == common.ProductionEnv {
		if primIOS, err = mergeMaps(config.Primary.Config.Common, config.Primary.Config.IOS); err != nil {
			err = xerrors.Errorf("unable to merge primary ios config")
			return
		}
		if primAndroid, err = mergeMaps(config.Primary.Config.Common, config.Primary.Config.Android); err != nil {
			err = xerrors.Errorf("unable to merge primary ios config")
			return
		}
	} else {
		if primIOS, err = mergeMaps(
			config.Primary.Config.Common,
			config.Primary.Config.IOS,
			config.Testing.Config.Common,
			config.Testing.Config.IOS,
		); err != nil {
			err = xerrors.Errorf("unable to merge primary ios config")
			return
		}
		if primAndroid, err = mergeMaps(
			config.Primary.Config.Common,
			config.Primary.Config.Android,
			config.Testing.Config.Common,
			config.Testing.Config.Android,
		); err != nil {
			err = xerrors.Errorf("unable to merge primary ios config")
			return
		}
	}
	if len(primIOS) == 0 {
		err = xerrors.Errorf("no settings in iOS config")
		return
	}
	if len(primAndroid) == 0 {
		err = xerrors.Errorf("no settings in Android config")
		return
	}
	if hasDebug {
		if debugIOS, err = mergeMaps(
			config.Primary.Config.Common,
			config.Primary.Config.IOS,
			config.Debug.Config.Common,
			config.Debug.Config.IOS); err != nil {
			err = xerrors.Errorf("unable to merge debug ios config: %w", err)
			return
		}
		if debugAndroid, err = mergeMaps(
			config.Primary.Config.Common,
			config.Primary.Config.Android,
			config.Debug.Config.Common,
			config.Debug.Config.Android); err != nil {
			err = xerrors.Errorf("unable to merge debug android config: %w", err)
			return
		}
	}
	return
}

func mergeMaps(maps ...map[string]interface{}) (map[string]interface{}, error) {
	result := make(map[string]interface{})
	var err error
	var initialized bool
	for i, patch := range maps {
		if patch == nil {
			continue
		}
		if !initialized {
			for key, value := range patch {
				result[key] = value
			}
			initialized = true
		} else {
			result, err = deepMerge(result, patch, "")
			if err != nil {
				return nil, xerrors.Errorf("unable to merge path %d", i)
			}
		}
	}
	return result, nil
}

func deepMerge(main map[string]interface{}, patch map[string]interface{}, path string) (map[string]interface{}, error) {
	if patch == nil {
		return main, nil
	}
	result := make(map[string]interface{})
	for key, value := range main {
		result[key] = value
	}
	for key, value := range patch {
		existing, exists := main[key]
		if exists {
			if castPatch, patchIsMap := value.(map[string]interface{}); patchIsMap {
				castSource, sourceIsMap := existing.(map[string]interface{})
				newPath := path + "/" + key
				if !sourceIsMap {
					return nil, xerrors.Errorf("unable to merge maps: types at '%s' do not match", newPath)
				}
				nested, err := deepMerge(castSource, castPatch, newPath)
				if err != nil {
					return nil, err
				}
				result[key] = nested
			} else {
				result[key] = value
			}
		} else {
			result[key] = value
		}
	}
	return result, nil
}
