package handler

import (
	"context"
	"fmt"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"
	"google.golang.org/grpc/status"

	system "a.yandex-team.ru/travel/app/backend/api/system/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/healthswitch"
)

type GRPCSystemHandler struct {
	healthServer     healthpb.HealthServer
	serviceName      string
	cancelFunc       func()
	shutdownInterval time.Duration
	shutdown         *healthswitch.Switch
	pingableServices map[string]Pingable
	environment      common.EnvType
}

type Pingable interface {
	Ping(context.Context) error
	ID() string
}

func NewGRPCSystemHandler(healthServer healthpb.HealthServer, serviceName string, cancelFunc context.CancelFunc,
	shutdownInterval time.Duration, shutdown *healthswitch.Switch, environment common.EnvType) *GRPCSystemHandler {
	return &GRPCSystemHandler{
		healthServer:     healthServer,
		serviceName:      serviceName,
		cancelFunc:       cancelFunc,
		shutdownInterval: shutdownInterval,
		shutdown:         shutdown,
		environment:      environment,
	}
}

func (h *GRPCSystemHandler) SetPingableServices(pingableServices ...Pingable) {
	h.pingableServices = make(map[string]Pingable, len(pingableServices))
	for _, s := range pingableServices {
		h.pingableServices[s.ID()] = s
	}
}

func (h *GRPCSystemHandler) Ping(ctx context.Context, req *system.PingReq) (*system.PingRsp, error) {
	if !h.shutdown.GetHealth() {
		return nil, status.Error(codes.Unavailable, "shutdown initialized")
	}
	for id, service := range h.pingableServices {
		err := service.Ping(ctx)
		if err != nil {
			return nil, status.Error(codes.Unavailable, fmt.Sprintf("%s unavailable: %s", id, err.Error()))
		}
	}
	rsp, err := h.healthServer.Check(ctx, &healthpb.HealthCheckRequest{Service: h.serviceName})
	if err != nil {
		return nil, err
	}
	if rsp.Status == healthpb.HealthCheckResponse_SERVING {
		return &system.PingRsp{Healthy: true}, nil
	} else {
		return nil, status.Error(codes.Unavailable, fmt.Sprintf("health status is %s", rsp.Status.String()))
	}
}

func (h *GRPCSystemHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		system.RegisterSystemAPIServer(server, h)
	}
}
