package handler

import (
	"context"
	"time"

	"google.golang.org/grpc"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/library/go/core/xerrors"
	systemlocalhost "a.yandex-team.ru/travel/app/backend/api/systemlocalhost/v1"
	"a.yandex-team.ru/travel/app/backend/internal/healthswitch"
)

type GRPCSystemLocalhostHandler struct {
	healthServer       healthpb.HealthServer
	serviceName        string
	cancelFunc         func()
	shutdownInterval   time.Duration
	shutdown           *healthswitch.Switch
	reloadableServices map[string]Reloadable
}

type Reloadable interface {
	Reload() error
	ID() string
}

func NewGRPCSystemLocalhostHandler(healthServer healthpb.HealthServer, serviceName string, cancelFunc context.CancelFunc,
	shutdownInterval time.Duration, shutdown *healthswitch.Switch) *GRPCSystemLocalhostHandler {
	return &GRPCSystemLocalhostHandler{
		healthServer:     healthServer,
		serviceName:      serviceName,
		cancelFunc:       cancelFunc,
		shutdownInterval: shutdownInterval,
		shutdown:         shutdown,
	}
}

func (h *GRPCSystemLocalhostHandler) SetReloadableServices(reloadableServices ...Reloadable) {
	h.reloadableServices = make(map[string]Reloadable, len(reloadableServices))
	for _, s := range reloadableServices {
		h.reloadableServices[s.ID()] = s
	}
}

func (h *GRPCSystemLocalhostHandler) Shutdown(ctx context.Context, req *systemlocalhost.ShutdownReq) (*systemlocalhost.ShutdownRsp, error) {
	h.shutdown.SetHealth(false)
	go func() {
		var toSleep time.Duration
		if req.ShutdownInMs != nil {
			toSleep = time.Duration(req.ShutdownInMs.Value) * time.Millisecond
		} else {
			toSleep = h.shutdownInterval
		}
		time.Sleep(toSleep)
		h.cancelFunc()
	}()
	return &systemlocalhost.ShutdownRsp{}, nil
}

func (h *GRPCSystemLocalhostHandler) Reload(ctx context.Context, req *systemlocalhost.ReloadReq) (*systemlocalhost.ReloadRsp, error) {
	var ids []string
	if len(req.Ids) == 0 {
		ids = make([]string, len(h.reloadableServices))
		i := 0
		for id := range h.reloadableServices {
			ids[i] = id
			i++
		}
	} else {
		ids = req.Ids
	}
	for _, id := range ids {
		var service Reloadable
		var exists bool
		if service, exists = h.reloadableServices[id]; !exists {
			return nil, xerrors.Errorf("no reloadable service with id '%s' is registered", id)
		}
		if err := service.Reload(); err != nil {
			return nil, xerrors.Errorf("unable to reload service '%s': %w", service.ID(), err)
		}
	}
	return &systemlocalhost.ReloadRsp{}, nil
}

func (h *GRPCSystemLocalhostHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		systemlocalhost.RegisterSystemLocalhostAPIServer(server, h)
	}
}
