package travelers

import (
	"embed"
	"encoding/json"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/app/backend/pkg/hashutils"
)

//go:embed admindata/*.json
var adminData embed.FS

type VisualFieldDefinition struct {
	InputMask string    `json:"input_mask,omitempty"`
	TankerKey string    `json:"tanker_key"`
	Type      FieldType `json:"type"`
}

type BackendFieldDefinition struct {
	Type         FieldType `json:"type"`
	DefaultValue string    `json:"default_value,omitempty"`
}

type GroupDefinition struct {
	TitleTankerKey    string   `json:"title_tanker_key,omitempty"`
	SubtitleTankerKey string   `json:"subtitle_tanker_key,omitempty"`
	FieldNames        []string `json:"field_names"`
}

type VisualEntity struct {
	TankerKey                string                            `json:"tanker_key,omitempty"`
	IconURL                  string                            `json:"icon_url,omitempty"`
	BackgroundColor          string                            `json:"background_color,omitempty"`
	VisualFieldDefinitions   map[string]VisualFieldDefinition  `json:"visual_field_definitions"`
	BackendFieldDefinitions  map[string]BackendFieldDefinition `json:"backend_field_definitions,omitempty"`
	CreationGroupDefinitions []GroupDefinition                 `json:"creation_group_definitions"`
	EditGroupDefinitions     []GroupDefinition                 `json:"edit_group_definitions"`
}

type TravelersAdminCache struct {
	visualPassenger        VisualEntity
	visualDocumentsByType  map[string]VisualEntity
	visualBonusCardsByType map[string]VisualEntity
	tag                    string
	logger                 log.Logger
}

func NewTravelersAdminCache(logger log.Logger) (*TravelersAdminCache, error) {
	pData, err := adminData.ReadFile("admindata/passenger.json")
	if err != nil {
		return nil, xerrors.Errorf("fail to load passenger data: %w", err)
	}
	var p VisualEntity
	err = json.Unmarshal(pData, &p)
	if err != nil {
		return nil, xerrors.Errorf("fail to load passenger data: %w", err)
	}

	docsData, err := adminData.ReadFile("admindata/documents.json")
	if err != nil {
		return nil, xerrors.Errorf("fail to load documents data: %w", err)
	}
	var docs map[string]VisualEntity
	err = json.Unmarshal(docsData, &docs)
	if err != nil {
		return nil, xerrors.Errorf("fail to load documents data: %w", err)
	}

	bonusData, err := adminData.ReadFile("admindata/bonus_cards.json")
	if err != nil {
		return nil, xerrors.Errorf("fail to load bonus cards data: %w", err)
	}
	var bonusCards map[string]VisualEntity
	err = json.Unmarshal(bonusData, &bonusCards)
	if err != nil {
		return nil, xerrors.Errorf("fail to load bonus cards data: %w", err)
	}

	a := TravelersAdminCache{
		logger:                 logger,
		visualPassenger:        p,
		visualDocumentsByType:  docs,
		visualBonusCardsByType: bonusCards,
	}

	err = a.validate()
	if err != nil {
		return nil, xerrors.Errorf("fail to validate: %w", err)
	}

	pTag, err := hashutils.GetSimpleHashFromStruct(p)
	if err != nil {
		return nil, xerrors.Errorf("fail to get passenger data hash: %w", err)
	}
	dTag, err := hashutils.GetSimpleHashFromStruct(docs)
	if err != nil {
		return nil, xerrors.Errorf("fail to get documents data hash: %w", err)
	}
	bcTag, err := hashutils.GetSimpleHashFromStruct(bonusCards)
	if err != nil {
		return nil, xerrors.Errorf("fail to get bonus cards data hash: %w", err)
	}
	tag, err := hashutils.GetSimpleHash(pTag + dTag + bcTag)
	if err != nil {
		return nil, xerrors.Errorf("fail to get final data hash: %w", err)
	}
	a.tag = tag
	return &a, nil
}

func (a *TravelersAdminCache) GetTag() string {
	return a.tag
}

func (a *TravelersAdminCache) GetVisualPassenger() VisualEntity {
	return a.visualPassenger
}

func (a *TravelersAdminCache) GetVisualDocumentsByType() map[string]VisualEntity {
	return a.visualDocumentsByType
}

func (a *TravelersAdminCache) GetVisualBonusCardsByType() map[string]VisualEntity {
	return a.visualBonusCardsByType
}

func (a *TravelersAdminCache) GetTankerKeys() []string {
	keys := make(map[string]struct{})
	for _, v := range a.visualPassenger.VisualFieldDefinitions {
		keys[v.TankerKey] = struct{}{}
	}
	for _, g := range a.visualPassenger.CreationGroupDefinitions {
		keys[g.TitleTankerKey] = struct{}{}
		keys[g.SubtitleTankerKey] = struct{}{}
	}
	for _, g := range a.visualPassenger.EditGroupDefinitions {
		keys[g.TitleTankerKey] = struct{}{}
		keys[g.SubtitleTankerKey] = struct{}{}
	}
	for _, e := range a.visualDocumentsByType {
		keys[e.TankerKey] = struct{}{}
		for _, v := range e.VisualFieldDefinitions {
			keys[v.TankerKey] = struct{}{}
		}
		for _, g := range e.CreationGroupDefinitions {
			keys[g.TitleTankerKey] = struct{}{}
			keys[g.SubtitleTankerKey] = struct{}{}
		}
		for _, g := range e.EditGroupDefinitions {
			keys[g.TitleTankerKey] = struct{}{}
			keys[g.SubtitleTankerKey] = struct{}{}
		}
	}
	for _, e := range a.visualBonusCardsByType {
		keys[e.TankerKey] = struct{}{}
		for _, v := range e.VisualFieldDefinitions {
			keys[v.TankerKey] = struct{}{}
		}
		for _, g := range e.CreationGroupDefinitions {
			keys[g.TitleTankerKey] = struct{}{}
			keys[g.SubtitleTankerKey] = struct{}{}
		}
		for _, g := range e.EditGroupDefinitions {
			keys[g.TitleTankerKey] = struct{}{}
			keys[g.SubtitleTankerKey] = struct{}{}
		}
	}
	res := make([]string, 0, len(keys))
	for k := range keys {
		if len(k) > 0 {
			res = append(res, k)
		}
	}
	return res
}

func (a *TravelersAdminCache) validate() error {
	err := validateEntity(a.visualPassenger)
	if err != nil {
		return xerrors.Errorf("invalid passenger data: %w", err)
	}

	for t, d := range a.visualDocumentsByType {
		err = validateEntity(d)
		if err != nil {
			return xerrors.Errorf("invalid document data for %s: %w", t, err)
		}
	}

	for t, bc := range a.visualBonusCardsByType {
		err = validateEntity(bc)
		if err != nil {
			return xerrors.Errorf("invalid document data for %s: %w", t, err)
		}
	}

	return nil
}

func validateEntity(e VisualEntity) error {
	actualFields := make(map[string]struct{})
	for _, g := range e.CreationGroupDefinitions {
		for _, f := range g.FieldNames {
			actualFields[f] = struct{}{}
		}
	}
	for _, g := range e.EditGroupDefinitions {
		for _, f := range g.FieldNames {
			actualFields[f] = struct{}{}
		}
	}
	for f := range actualFields {
		_, visualOk := e.VisualFieldDefinitions[f]
		_, backendOk := e.BackendFieldDefinitions[f]
		if !visualOk && !backendOk {
			return xerrors.Errorf("no field definition for field %s", f)
		}
	}
	return nil
}
