package handler

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	"github.com/golang/protobuf/proto"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/anypb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	travelersAPI "a.yandex-team.ru/travel/app/backend/api/travelers/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelersclient"
	"a.yandex-team.ru/travel/app/backend/internal/travelers"
)

const l10nKeyset = "backend-travelers"

type GRPCTravelersHandler struct {
	logger          log.Logger
	travelersClient travelers.Client
	travelersCache  travelers.Cache
	adminCache      travelers.AdminCache
	l10nService     travelers.L10nService
}

func NewGRPCTravelersHandler(
	l log.Logger,
	tc travelers.Client,
	travelersCache travelers.Cache,
	adminCache travelers.AdminCache,
	l10nService travelers.L10nService,
) *GRPCTravelersHandler {
	h := GRPCTravelersHandler{
		logger:          l,
		travelersClient: tc,
		travelersCache:  travelersCache,
		adminCache:      adminCache,
		l10nService:     l10nService,
	}
	return &h
}

func (h *GRPCTravelersHandler) GetL10NKeysetName() string {
	return l10nKeyset
}

func (h *GRPCTravelersHandler) GetL10NKeys() []string {
	return h.adminCache.GetTankerKeys()
}

func (h *GRPCTravelersHandler) ListFields(ctx context.Context, req *travelersAPI.ListFieldsReq) (*travelersAPI.ListFieldsRsp, error) {
	lang := common.GetLanguage(ctx)
	keyset, err := h.l10nService.Get(l10nKeyset, lang)
	if err != nil {
		if xerrors.Is(err, l10n.UnsupportedLanguage) {
			return nil, status.Error(codes.InvalidArgument, fmt.Sprintf("unsupported language '%s'", lang))
		}
		msg := fmt.Sprintf("no keyset for %s, %s", l10nKeyset, lang)
		h.logger.Error(msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}

	fd := h.travelersCache.GetFieldsData()

	newTag := strings.Join([]string{fd.Tag, h.adminCache.GetTag(), keyset.Tag}, "_")
	if req.Tag == newTag {
		return &travelersAPI.ListFieldsRsp{
			Tag: req.Tag,
		}, nil
	}
	passenger, err := travelers.ConvertPassengerFieldsToProto(fd.PassengerFields, h.adminCache.GetVisualPassenger(), keyset.Keys)
	if err != nil {
		msg := "error building passenger fields"
		h.logger.Error(msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}
	docs, err := travelers.ConvertDocumentTypesToProto(fd.DocumentTypes, h.adminCache.GetVisualDocumentsByType(), keyset.Keys)
	if err != nil {
		msg := "error building documents fields"
		h.logger.Error(msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}
	bonusCards, err := travelers.ConvertBonusCardTypesToProto(fd.BonusCardTypes, h.adminCache.GetVisualBonusCardsByType(), keyset.Keys)
	if err != nil {
		msg := "error building bonus cards fields"
		h.logger.Error(msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}

	result := travelersAPI.ListFieldsRsp{
		Tag:             newTag,
		PassengerGroups: passenger,
		DocumentTypes:   docs,
		BonusCardTypes:  bonusCards,
	}
	return &result, nil
}

func (h *GRPCTravelersHandler) ListPassengers(ctx context.Context, req *travelersAPI.ListPassengersReq) (*travelersAPI.ListPassengersRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}
	if res, err := h.travelersClient.ListPassengers(ctx, strconv.FormatUint(auth.User.ID, 10), req.IncludeDetails, req.IncludeDetails); err != nil {
		return nil, convertError(err, h.logger)
	} else {
		return travelers.ConvertPassengerListToProto(res, h.adminCache), nil
	}
}

func (h *GRPCTravelersHandler) GetPassenger(ctx context.Context, req *travelersAPI.GetPassengerReq) (*travelersAPI.GetPassengerRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}

	var passengerInfo *travelersclient.Passenger
	if passengerInfo, err = h.travelersClient.GetPassenger(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId, req.IncludeDetails, req.IncludeDetails); err != nil {
		return nil, convertError(xerrors.Errorf("unable to get passenger info: %w", err), h.logger)
	}
	resp := travelersAPI.GetPassengerRsp{
		Passenger: travelers.ConvertPassengerToProto(passengerInfo, h.adminCache),
	}
	return &resp, nil
}

func (h *GRPCTravelersHandler) CreatePassenger(ctx context.Context, req *travelersAPI.CreatePassengerReq) (*travelersAPI.CreatePassengerRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}

	uid := strconv.FormatUint(auth.User.ID, 10)
	if err = h.createTravelerIfNotExists(ctx, uid); err != nil {
		return nil, err
	}

	r := travelers.ConvertProtoToPassengerRequest(req.Passenger, h.adminCache)
	var passenger *travelersclient.Passenger
	if passenger, err = h.travelersClient.CreatePassenger(ctx, uid, r); err != nil {
		if badRequest, ok := travelers.PassengerErrorAsBadRequest(err); ok {
			st, _ := status.New(codes.InvalidArgument, "PassengerValidationFailed").WithDetails(badRequest)
			return nil, st.Err()
		}
		return nil, convertError(err, h.logger)
	}

	var hasErrors bool
	var chanErrs []error
	var documentErrors []*travelersAPI.NestedEntityCreationError
	var cardErrors []*travelersAPI.NestedEntityCreationError
	if len(req.Passenger.Documents) > 0 || len(req.Passenger.BonusCards) > 0 {
		documentChannels := make([]chan createDocumentResult, 0, len(req.Passenger.Documents))
		for i, d := range req.Passenger.Documents {
			doc := travelers.ConvertProtoToCreateOrUpdateDocument(d, h.adminCache)
			documentChannels = append(documentChannels, h.createDocumentAsync(ctx, uid, passenger.ID, doc, i))
		}
		bonusCardChannels := make([]chan createBonusCardResult, 0, len(req.Passenger.BonusCards))
		for i, bc := range req.Passenger.BonusCards {
			bonusCard := travelers.ConvertProtoToEditableBonusCard(bc, h.adminCache)
			bonusCardChannels = append(bonusCardChannels, h.createBonusCardAsync(ctx, uid, passenger.ID, bonusCard, i))
		}

		for _, documentChan := range documentChannels {
			dr, ok := <-documentChan
			if !ok {
				hasErrors = true
				chanErrs = append(chanErrs, fmt.Errorf("can't get document from channel"))
			} else if dr.ErrProto != nil {
				hasErrors = true
				documentErrors = append(documentErrors, dr.ErrProto)
			} else {
				passenger.Documents = append(passenger.Documents, *dr.Document)
			}
		}
		for _, bonusCardChan := range bonusCardChannels {
			bcr, ok := <-bonusCardChan
			if !ok {
				hasErrors = true
				chanErrs = append(chanErrs, fmt.Errorf("can't get bonus card from channel"))
			} else if bcr.ErrProto != nil {
				hasErrors = true
				cardErrors = append(cardErrors, bcr.ErrProto)
			} else {
				passenger.BonusCards = append(passenger.BonusCards, *bcr.BonusCard)
			}
		}
	}
	if hasErrors {
		// Ошибки чтения из каналов: могут случаться только при остановке контекста, в остальных случаях их не будет.
		for _, err := range chanErrs {
			h.logger.Error("channel error occurred", log.Error(err))
		}
		// Если сохранили не все, то нужно вернуть ответ с тем, что сохранили.
		// Если вернем ошибку, то тело будет проигнорировано
		// Поэтому ошибки только логируем.
		msg := fmt.Sprintf(
			"not all data saved, saved %d/%d documents, %d/%d bonus cards",
			len(passenger.Documents),
			len(req.Passenger.Documents),
			len(passenger.BonusCards),
			len(req.Passenger.BonusCards),
		)
		h.logger.Warn(msg)
	}

	return &travelersAPI.CreatePassengerRsp{
		Passenger:                travelers.ConvertPassengerToProto(passenger, h.adminCache),
		NotSavedDocumentsCount:   (uint32)(len(req.Passenger.Documents) - len(passenger.Documents)),
		NotSavedBonusCardsCount:  (uint32)(len(req.Passenger.BonusCards) - len(passenger.BonusCards)),
		DocumentCreationErrors:   documentErrors,
		BonusCardsCreationErrors: cardErrors,
	}, nil
}

func (h *GRPCTravelersHandler) createDocumentAsync(ctx context.Context, uid, passengerID string, doc *travelersclient.CreateOrUpdateDocumentRequest, index int) chan createDocumentResult {
	resChan := make(chan createDocumentResult)
	go func() {
		defer close(resChan)

		r, err := h.travelersClient.CreateDocument(ctx, uid, passengerID, doc)
		var errProto *travelersAPI.NestedEntityCreationError
		if err != nil {
			var detail anypb.Any
			if p, ok := travelers.DocumentErrorAsBadRequest(err); ok {
				h.logger.Info("Validation error for nested document", log.Error(err))
				_ = detail.MarshalFrom(p)
			} else {
				h.logger.Info("Unexpected error for nested document", log.Error(err))
				_ = detail.MarshalFrom(&travelersAPI.OtherTravelersError{ErrorMessage: err.Error()})
			}
			errProto = &travelersAPI.NestedEntityCreationError{
				Index:   uint32(index),
				Details: []*anypb.Any{&detail},
			}
		}
		select {
		case resChan <- createDocumentResult{Document: r, ErrProto: errProto}:
			return
		case <-ctx.Done():
			return
		}
	}()
	return resChan
}

func (h *GRPCTravelersHandler) createBonusCardAsync(ctx context.Context, uid, passengerID string, bc *travelersclient.EditableBonusCard, index int) chan createBonusCardResult {
	resChan := make(chan createBonusCardResult)
	go func() {
		defer close(resChan)

		r, err := h.travelersClient.CreateBonusCard(ctx, uid, passengerID, bc)
		var errProto *travelersAPI.NestedEntityCreationError
		if err != nil {
			var detail anypb.Any
			if p, ok := travelers.CardErrorAsBadRequest(err); ok {
				h.logger.Info("Validation error for nested bonus card", log.Error(err))
				_ = detail.MarshalFrom(p)
			} else {
				h.logger.Info("Unexpected error for nested bonus card", log.Error(err))
				_ = detail.MarshalFrom(&travelersAPI.OtherTravelersError{ErrorMessage: err.Error()})
			}
			errProto = &travelersAPI.NestedEntityCreationError{
				Index:   uint32(index),
				Details: []*anypb.Any{&detail},
			}
		}
		select {
		case resChan <- createBonusCardResult{r, errProto}:
			return
		case <-ctx.Done():
			return
		}
	}()
	return resChan
}

func (h *GRPCTravelersHandler) UpdatePassenger(ctx context.Context, req *travelersAPI.UpdatePassengerReq) (*travelersAPI.UpdatePassengerRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}
	r := travelers.ConvertProtoToPassengerRequest(req.Passenger, h.adminCache)
	var pass *travelersclient.Passenger
	if pass, err = h.travelersClient.UpdatePassenger(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId, r); err != nil {
		if badRequest, ok := travelers.PassengerErrorAsBadRequest(err); ok {
			st, _ := status.New(codes.InvalidArgument, "PassengerValidationFailed").WithDetails(badRequest)
			return nil, st.Err()
		}
		return nil, convertError(err, h.logger)
	}
	return &travelersAPI.UpdatePassengerRsp{Passenger: travelers.ConvertPassengerToProto(pass, h.adminCache)}, nil
}

func (h *GRPCTravelersHandler) DeletePassenger(ctx context.Context, req *travelersAPI.DeletePassengerReq) (*travelersAPI.DeletePassengerRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}
	if err = h.travelersClient.DeletePassenger(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId); err != nil {
		return nil, convertError(err, h.logger)
	}
	return &travelersAPI.DeletePassengerRsp{}, nil
}

func (h *GRPCTravelersHandler) ListPassengerDocuments(ctx context.Context, req *travelersAPI.ListDocumentsReq) (*travelersAPI.ListDocumentsRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}
	if res, err := h.travelersClient.ListDocuments(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId); err != nil {
		return nil, convertError(err, h.logger)
	} else {
		return &travelersAPI.ListDocumentsRsp{Documents: travelers.ConvertDocumentListToProto(res, h.adminCache)}, nil
	}
}

func (h *GRPCTravelersHandler) GetPassengerDocument(ctx context.Context, req *travelersAPI.GetDocumentReq) (*travelersAPI.GetDocumentRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}
	if res, err := h.travelersClient.GetDocument(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId, req.DocumentId); err != nil {
		return nil, convertError(err, h.logger)
	} else {
		resp := travelersAPI.GetDocumentRsp{Document: travelers.ConvertDocumentToProto(res, h.adminCache)}
		return &resp, nil
	}
}

func (h *GRPCTravelersHandler) CreateDocument(ctx context.Context, req *travelersAPI.CreateDocumentReq) (*travelersAPI.CreateDocumentRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}
	d := travelers.ConvertProtoToCreateOrUpdateDocument(req.Document, h.adminCache)
	if res, err := h.travelersClient.CreateDocument(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId, d); err != nil {
		if badRequest, ok := travelers.DocumentErrorAsBadRequest(err); ok {
			st, _ := status.New(codes.InvalidArgument, "DocumentValidationFailed").WithDetails(badRequest)
			return nil, st.Err()
		}
		return nil, convertError(err, h.logger)
	} else {
		resp := travelersAPI.CreateDocumentRsp{Document: travelers.ConvertDocumentToProto(res, h.adminCache)}
		return &resp, nil
	}
}

func (h *GRPCTravelersHandler) UpdateDocument(ctx context.Context, req *travelersAPI.UpdateDocumentReq) (*travelersAPI.UpdateDocumentRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}
	d := travelers.ConvertProtoToCreateOrUpdateDocument(req.Document, h.adminCache)
	if res, err := h.travelersClient.UpdateDocument(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId, req.DocumentId, d); err != nil {
		if badRequest, ok := travelers.DocumentErrorAsBadRequest(err); ok {
			st, _ := status.New(codes.InvalidArgument, "DocumentValidationFailed").WithDetails(badRequest)
			return nil, st.Err()
		}
		return nil, convertError(err, h.logger)
	} else {
		resp := travelersAPI.UpdateDocumentRsp{Document: travelers.ConvertDocumentToProto(res, h.adminCache)}
		return &resp, nil
	}
}

func (h *GRPCTravelersHandler) DeleteDocument(ctx context.Context, req *travelersAPI.DeleteDocumentReq) (*travelersAPI.DeleteDocumentRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}
	if err := h.travelersClient.DeleteDocument(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId, req.DocumentId); err != nil {
		return nil, convertError(err, h.logger)
	} else {
		return &travelersAPI.DeleteDocumentRsp{}, nil
	}
}

func (h *GRPCTravelersHandler) ListBonusCards(ctx context.Context, req *travelersAPI.ListBonusCardsReq) (*travelersAPI.ListBonusCardsRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}

	resp, err := h.travelersClient.ListBonusCards(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId)
	if err != nil {
		return nil, convertError(err, h.logger)
	}

	return &travelersAPI.ListBonusCardsRsp{BonusCards: travelers.ConvertBonusCardsToProto(resp, h.adminCache)}, nil
}

func (h *GRPCTravelersHandler) GetBonusCard(ctx context.Context, req *travelersAPI.GetBonusCardReq) (*travelersAPI.GetBonusCardRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}

	resp, err := h.travelersClient.GetBonusCard(ctx, strconv.FormatUint(auth.User.ID, 10), req.PassengerId, req.BonusCardId)
	if err != nil {
		return nil, convertError(err, h.logger)
	}

	return &travelersAPI.GetBonusCardRsp{BonusCard: travelers.ConvertBonusCardToProto(resp, h.adminCache)}, nil
}

func (h *GRPCTravelersHandler) CreateBonusCard(ctx context.Context, req *travelersAPI.CreateBonusCardReq) (*travelersAPI.CreateBonusCardRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}

	uid := strconv.FormatUint(auth.User.ID, 10)
	bonusCard := travelers.ConvertProtoToEditableBonusCard(req.BonusCard, h.adminCache)
	resp, err := h.travelersClient.CreateBonusCard(ctx, uid, req.PassengerId, bonusCard)
	if err != nil {
		if badRequest, ok := travelers.CardErrorAsBadRequest(err); ok {
			st, _ := status.New(codes.InvalidArgument, "CardValidationFailed").WithDetails(badRequest)
			return nil, st.Err()
		}
		return nil, convertError(err, h.logger)
	}

	return &travelersAPI.CreateBonusCardRsp{BonusCard: travelers.ConvertBonusCardToProto(resp, h.adminCache)}, nil
}

func (h *GRPCTravelersHandler) UpdateBonusCard(ctx context.Context, req *travelersAPI.UpdateBonusCardReq) (*travelersAPI.UpdateBonusCardRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}

	uid := strconv.FormatUint(auth.User.ID, 10)
	bonusCard := travelers.ConvertProtoToEditableBonusCard(req.BonusCard, h.adminCache)
	resp, err := h.travelersClient.UpdateBonusCard(ctx, uid, req.PassengerId, req.BonusCardId, bonusCard)
	if err != nil {
		if badRequest, ok := travelers.CardErrorAsBadRequest(err); ok {
			st, _ := status.New(codes.InvalidArgument, "CardValidationFailed").WithDetails(badRequest)
			return nil, st.Err()
		}
		return nil, convertError(err, h.logger)
	}

	return &travelersAPI.UpdateBonusCardRsp{BonusCard: travelers.ConvertBonusCardToProto(resp, h.adminCache)}, nil
}

func (h *GRPCTravelersHandler) DeleteBonusCard(ctx context.Context, req *travelersAPI.DeleteBonusCardReq) (*travelersAPI.DeleteBonusCardRsp, error) {
	auth, err := common.RequireAuth(ctx)
	if err != nil {
		return nil, err
	}

	uid := strconv.FormatUint(auth.User.ID, 10)
	err = h.travelersClient.DeleteBonusCard(ctx, uid, req.PassengerId, req.BonusCardId)
	if err != nil {
		return nil, convertError(err, h.logger)
	}
	return &travelersAPI.DeleteBonusCardRsp{}, nil
}

func (h *GRPCTravelersHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		travelersAPI.RegisterTravelersAPIServer(server, h)
	}
}

func (h *GRPCTravelersHandler) createTravelerIfNotExists(ctx context.Context, uid string) error {
	_, err := h.travelersClient.GetTraveler(ctx, uid)
	if err == nil {
		return nil
	}

	var httpErr travelersclient.StatusError
	if xerrors.As(err, &httpErr) && httpErr.Status == 404 {
		traveler := travelersclient.EditableTraveler{
			Email:           "",
			Phone:           "",
			PhoneAdditional: "",
			Agree:           true,
		}
		if _, err = h.travelersClient.CreateOrUpdateTraveler(ctx, uid, &traveler); err != nil {
			return convertError(err, h.logger)
		}
		return nil
	}
	return convertError(err, h.logger)
}

type createDocumentResult struct {
	Document *travelersclient.Document
	ErrProto *travelersAPI.NestedEntityCreationError
}

type createBonusCardResult struct {
	BonusCard *travelersclient.BonusCard
	ErrProto  *travelersAPI.NestedEntityCreationError
}

func convertError(err error, logger log.Logger) error {
	var httpErr travelersclient.StatusError
	if xerrors.As(err, &httpErr) {
		logger.Error("got http error from travelers", log.Int("status", httpErr.Status), log.Any("response", httpErr.Response))
		switch {
		case httpErr.Status == 400:
			return status.Error(codes.InvalidArgument, "bad request")
		case httpErr.Status == 403:
			return status.Error(codes.Unauthenticated, "unauthenticated")
		case httpErr.Status == 404:
			return status.Error(codes.NotFound, "not found")
		case httpErr.Status >= 500:
			return addProtoDetailsToStatus(status.New(codes.Unknown, "invalid travellers response"),
				&travelersAPI.OtherTravelersError{ErrorMessage: err.Error()})
		default:
			return addProtoDetailsToStatus(status.New(codes.Unknown, "unexpected error"), &travelersAPI.OtherTravelersError{ErrorMessage: err.Error()})
		}
	} else {
		switch {
		case xerrors.Is(err, travelersclient.NoServiceTicketError):
			return status.Error(codes.Unauthenticated, "no service ticket error")
		case xerrors.Is(err, travelersclient.NoUserTicketError):
			return status.Error(codes.Unauthenticated, "no user ticket error")
		case xerrors.Is(err, travelersclient.ResponseError):
			return status.Error(codes.Unknown, "can not get response")
		default:
			return addProtoDetailsToStatus(status.New(codes.Unknown, "invalid travellers response"),
				&travelersAPI.OtherTravelersError{ErrorMessage: err.Error()})
		}
	}
}

func addProtoDetailsToStatus(st *status.Status, details proto.Message) error {
	s, e := st.WithDetails(details)
	if e != nil {
		return st.Err()
	} else {
		return s.Err()
	}
}
