package travelers

import (
	"context"
	"strconv"
	"time"

	"google.golang.org/genproto/googleapis/rpc/errdetails"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/xerrors"
	travelersAPI "a.yandex-team.ru/travel/app/backend/api/travelers/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelersclient"
)

type FieldType string

const (
	FieldTypeString  FieldType = "string"
	FieldTypeDate    FieldType = "date"
	FieldTypeCountry FieldType = "country"
	FieldTypeGender  FieldType = "gender"
	FieldTypeBool    FieldType = "bool"
)

var typeToProto = map[FieldType]travelersAPI.FieldType{
	FieldTypeString:  travelersAPI.FieldType_FIELD_TYPE_STRING,
	FieldTypeDate:    travelersAPI.FieldType_FIELD_TYPE_DATE,
	FieldTypeCountry: travelersAPI.FieldType_FIELD_TYPE_COUNTRY,
	FieldTypeGender:  travelersAPI.FieldType_FIELD_TYPE_GENDER,
	FieldTypeBool:    travelersAPI.FieldType_FIELD_TYPE_BOOL,
}

var passengerFieldNameToTravelersAPIName = map[string]string{
	"title":                       "title",
	"gender":                      "gender",
	"birth_date":                  "birth_date",
	"itn":                         "itn",
	"phone":                       "phone",
	"email":                       "email",
	"train_notifications_enabled": "train_notifications_enabled",
}

var passengerFieldNameToTravelersAPINameInverted = invertMap(passengerFieldNameToTravelersAPIName)

var documentFieldNameToTravelersAPIName = map[string]string{
	"title":              "title",
	"number":             "number",
	"first_name":         "first_name",
	"middle_name":        "middle_name",
	"last_name":          "last_name",
	"first_name_en":      "first_name_en",
	"middle_name_en":     "middle_name_en",
	"last_name_en":       "last_name_en",
	"issue_date":         "issue_date",
	"expiration_date":    "expiration_date",
	"citizenship_geo_id": "citizenship",
}

var documentFieldNameToTravelersAPINameInverted = invertMap(documentFieldNameToTravelersAPIName)

var bonusCardFieldNameToTravelersAPIName = map[string]string{
	"title":  "title",
	"number": "number",
}

var bonusCardFieldNameToTravelersAPINameInverted = invertMap(bonusCardFieldNameToTravelersAPIName)

func ConvertPassengerFieldsToProto(
	pf map[string]FieldSettings,
	visualPassenger VisualEntity,
	translations map[string]string,
) (*travelersAPI.PassengerGroups, error) {
	creationGroups, err := buildResultGroups(pf, visualPassenger.VisualFieldDefinitions, visualPassenger.CreationGroupDefinitions, translations)
	if err != nil {
		return nil, xerrors.Errorf("error building passenger fields: %w", err)
	}
	editGroups, err := buildResultGroups(pf, visualPassenger.VisualFieldDefinitions, visualPassenger.EditGroupDefinitions, translations)
	if err != nil {
		return nil, xerrors.Errorf("error building passenger fields: %w", err)
	}
	return &travelersAPI.PassengerGroups{
		CreationGroups: creationGroups,
		EditGroups:     editGroups,
	}, nil
}

func ConvertDocumentTypesToProto(
	dts []DocumentType,
	visualDocumentsByType map[string]VisualEntity,
	translations map[string]string,
) ([]*travelersAPI.DocumentType, error) {
	documentTypes := make([]*travelersAPI.DocumentType, 0, len(dts))
	for _, dt := range dts {
		visualDocument, ok := visualDocumentsByType[dt.Type]
		if !ok {
			continue
		}
		translation, ok := translations[visualDocument.TankerKey]
		if !ok {
			return nil, xerrors.Errorf("no translation for %s", visualDocument.TankerKey)
		}
		creationGroups, err := buildResultGroups(dt.FieldsSettings, visualDocument.VisualFieldDefinitions, visualDocument.CreationGroupDefinitions, translations)
		if err != nil {
			return nil, xerrors.Errorf("error building document types: %w", err)
		}
		editGroups, err := buildResultGroups(dt.FieldsSettings, visualDocument.VisualFieldDefinitions, visualDocument.EditGroupDefinitions, translations)
		if err != nil {
			return nil, xerrors.Errorf("error building document types: %w", err)
		}
		value := travelersAPI.DocumentType{
			Type:           dt.Type,
			IconUrl:        visualDocument.IconURL,
			Title:          translation,
			CreationGroups: creationGroups,
			EditGroups:     editGroups,
		}
		documentTypes = append(documentTypes, &value)
	}

	return documentTypes, nil
}

func ConvertBonusCardTypesToProto(
	bcts []BonusCardType,
	visualBonusCardsByType map[string]VisualEntity,
	translations map[string]string,
) ([]*travelersAPI.BonusCardType, error) {
	bonusCardTypes := make([]*travelersAPI.BonusCardType, 0, len(bcts))
	for _, bct := range bcts {
		visualBonusCard, ok := visualBonusCardsByType[bct.Type]
		if !ok {
			continue
		}
		translation, ok := translations[visualBonusCard.TankerKey]
		if !ok {
			return nil, xerrors.Errorf("no translation for %s", visualBonusCard.TankerKey)
		}
		creationGroups, err := buildResultGroups(bct.FieldsSettings, visualBonusCard.VisualFieldDefinitions, visualBonusCard.CreationGroupDefinitions, translations)
		if err != nil {
			return nil, xerrors.Errorf("error building bonus card types: %w", err)
		}
		editGroups, err := buildResultGroups(bct.FieldsSettings, visualBonusCard.VisualFieldDefinitions, visualBonusCard.EditGroupDefinitions, translations)
		if err != nil {
			return nil, xerrors.Errorf("error building bonus card types: %w", err)
		}
		value := travelersAPI.BonusCardType{
			Type:            bct.Type,
			Title:           translation,
			CreationGroups:  creationGroups,
			EditGroups:      editGroups,
			IconUrl:         visualBonusCard.IconURL,
			BackgroundColor: visualBonusCard.BackgroundColor,
		}
		bonusCardTypes = append(bonusCardTypes, &value)
	}

	return bonusCardTypes, nil
}

func buildResultGroups(
	fieldsSettings map[string]FieldSettings,
	visualFields map[string]VisualFieldDefinition,
	groups []GroupDefinition,
	translations map[string]string,
) ([]*travelersAPI.Group, error) {
	resultGroups := make([]*travelersAPI.Group, 0, len(groups))
	for _, group := range groups {
		fields := group.FieldNames
		resultFields := make([]*travelersAPI.Field, 0, len(fields))
		for _, field := range fields {
			tfs, ok := fieldsSettings[field]
			if !ok {
				continue
			}
			visual, ok := visualFields[field]
			if !ok {
				continue
			}
			fieldType, ok := typeToProto[visual.Type]
			if !ok {
				continue
			}
			translation, ok := translations[visual.TankerKey]
			if !ok {
				return nil, xerrors.Errorf("no translation for %s", visual.TankerKey)
			}
			cf := travelersAPI.Field{
				Name:     field,
				Type:     fieldType,
				Title:    translation,
				Required: tfs.Required,
			}
			if len(tfs.Regex) > 0 {
				cf.OptionalRegex = &travelersAPI.Field_Regex{Regex: tfs.Regex}
			}
			if len(visual.InputMask) > 0 {
				cf.OptionalInputMask = &travelersAPI.Field_InputMask{InputMask: visual.InputMask}
			}
			resultFields = append(resultFields, &cf)
		}
		g := travelersAPI.Group{
			Fields: resultFields,
		}
		if len(group.TitleTankerKey) > 0 {
			var ok bool
			g.Title, ok = translations[group.TitleTankerKey]
			if !ok {
				return nil, xerrors.Errorf("no translation for %s", group.TitleTankerKey)
			}
		}
		if len(group.SubtitleTankerKey) > 0 {
			var ok bool
			g.Subtitle, ok = translations[group.SubtitleTankerKey]
			if !ok {
				return nil, xerrors.Errorf("no translation for %s", group.SubtitleTankerKey)
			}
		}
		resultGroups = append(resultGroups, &g)
	}

	return resultGroups, nil
}

func ConvertPassengerToProto(p *travelersclient.Passenger, adminCache AdminCache) *travelersAPI.Passenger {
	res := travelersAPI.Passenger{
		Id:        p.ID,
		CreatedAt: timestamppb.New(p.CreatedAt.Time),
		UpdatedAt: timestamppb.New(p.UpdatedAt.Time),
		Fields:    makeFieldsFromRaw(p.Fields, adminCache.GetVisualPassenger(), passengerFieldNameToTravelersAPIName),
	}
	if len(p.Documents) > 0 {
		res.Documents = ConvertDocumentListToProto(p.Documents, adminCache)
	}
	if len(p.BonusCards) > 0 {
		res.BonusCards = ConvertBonusCardsToProto(p.BonusCards, adminCache)
	}
	return &res
}

func ConvertPassengerListToProto(ps []travelersclient.Passenger, adminCache AdminCache) *travelersAPI.ListPassengersRsp {
	res := travelersAPI.ListPassengersRsp{
		Passengers: make([]*travelersAPI.Passenger, len(ps)),
	}
	for i, p := range ps {
		pi := ConvertPassengerToProto(&p, adminCache)
		res.Passengers[i] = pi
	}
	return &res
}

func ConvertProtoToPassengerRequest(proto *travelersAPI.EditablePassenger, adminCache AdminCache) *travelersclient.CreateOrUpdatePassengerRequest {
	req := travelersclient.CreateOrUpdatePassengerRequest{}
	fields := convertProtoToFields(
		proto.Fields,
		adminCache.GetVisualPassenger(),
		passengerFieldNameToTravelersAPIName,
	)
	if len(fields) > 0 {
		req.Fields = fields
	}
	return &req
}

func ConvertProtoToCreateOrUpdateDocument(proto *travelersAPI.EditableDocument, adminCache AdminCache) *travelersclient.CreateOrUpdateDocumentRequest {
	doc := travelersclient.CreateOrUpdateDocumentRequest{
		Type: proto.Type,
	}
	docVisualEntity, ok := adminCache.GetVisualDocumentsByType()[proto.Type]
	if ok {
		fields := convertProtoToFields(
			proto.Fields,
			docVisualEntity,
			documentFieldNameToTravelersAPIName,
		)
		if len(fields) > 0 {
			doc.Fields = fields
		}
	}
	return &doc
}

func ConvertDocumentListToProto(ds []travelersclient.Document, adminCache AdminCache) []*travelersAPI.Document {
	res := make([]*travelersAPI.Document, len(ds))
	for i, d := range ds {
		res[i] = ConvertDocumentToProto(&d, adminCache)
	}
	return res
}

func ConvertDocumentToProto(d *travelersclient.Document, adminCache AdminCache) *travelersAPI.Document {
	document := travelersAPI.Document{
		Id:          d.ID,
		PassengerId: d.PassengerID,
		CreatedAt:   timestamppb.New(d.CreatedAt.Time),
		UpdatedAt:   timestamppb.New(d.UpdatedAt.Time),
		Type:        d.Type,
	}
	docVisualEntity, ok := adminCache.GetVisualDocumentsByType()[d.Type]
	if ok {
		fields := makeFieldsFromRaw(d.Fields, docVisualEntity, documentFieldNameToTravelersAPIName)
		if len(fields) > 0 {
			document.Fields = fields
		}
	}
	return &document
}

func ConvertBonusCardsToProto(bcs []travelersclient.BonusCard, adminCache AdminCache) []*travelersAPI.BonusCard {
	bonusCards := make([]*travelersAPI.BonusCard, len(bcs))
	for i, bc := range bcs {
		bonusCards[i] = ConvertBonusCardToProto(&bc, adminCache)
	}
	return bonusCards
}

func ConvertBonusCardToProto(bc *travelersclient.BonusCard, adminCache AdminCache) *travelersAPI.BonusCard {
	card := travelersAPI.BonusCard{
		Id:          bc.ID,
		PassengerId: bc.PassengerID,
		Type:        bc.Type,
		CreatedAt:   timestamppb.New(bc.CreatedAt.Time),
		UpdatedAt:   timestamppb.New(bc.UpdatedAt.Time),
	}
	bcVisualEntity, ok := adminCache.GetVisualBonusCardsByType()[bc.Type]
	if ok {
		fields := makeFieldsFromRaw(bc.Fields, bcVisualEntity, bonusCardFieldNameToTravelersAPIName)
		if len(fields) > 0 {
			card.Fields = fields
		}
	}
	return &card
}

func ConvertProtoToEditableBonusCard(bc *travelersAPI.EditableBonusCard, adminCache AdminCache) *travelersclient.EditableBonusCard {
	result := travelersclient.EditableBonusCard{
		Type: bc.Type,
	}
	visualBC, ok := adminCache.GetVisualBonusCardsByType()[bc.Type]
	if !ok {
		return &result
	}
	fields := convertProtoToFields(bc.Fields, visualBC, bonusCardFieldNameToTravelersAPIName)
	if len(fields) > 0 {
		result.Fields = fields
	}
	return &result
}

func PassengerErrorAsBadRequest(source error) (*errdetails.BadRequest, bool) {
	return errAsBadRequest(source, passengerFieldNameToTravelersAPINameInverted)
}

func DocumentErrorAsBadRequest(source error) (*errdetails.BadRequest, bool) {
	return errAsBadRequest(source, documentFieldNameToTravelersAPINameInverted)
}

func CardErrorAsBadRequest(source error) (*errdetails.BadRequest, bool) {
	return errAsBadRequest(source, bonusCardFieldNameToTravelersAPINameInverted)
}

func errAsBadRequest(source error, fieldNameMap map[string]string) (*errdetails.BadRequest, bool) {
	var validationError travelersclient.ValidationError
	if xerrors.As(source, &validationError) {
		msg := errdetails.BadRequest{}
		for f, errMsg := range validationError.FieldErrors {
			var fieldName string
			if _, exists := fieldNameMap[f]; exists {
				fieldName = fieldNameMap[f]
			} else {
				fieldName = f
			}
			msg.FieldViolations = append(msg.FieldViolations, &errdetails.BadRequest_FieldViolation{
				Field:       fieldName,
				Description: errMsg,
			})
		}
		return &msg, true
	}
	return nil, false
}

func convertProtoToFields(
	fields map[string]*travelersAPI.FieldValue,
	visualEntity VisualEntity,
	fieldNameToTravelersAPIName map[string]string,
) map[string]interface{} {
	result := make(map[string]interface{})
	for fieldName, apiName := range fieldNameToTravelersAPIName {
		backendField, hasBackendField := visualEntity.BackendFieldDefinitions[fieldName]
		if hasBackendField {
			v := getTravelersAPIValueFromString(backendField.DefaultValue, backendField.Type)
			if v != nil {
				result[apiName] = v
			}
		} else {
			fieldValue, ok := fields[fieldName]
			if !ok {
				continue
			}
			visualField, hasVisualField := visualEntity.VisualFieldDefinitions[fieldName]
			if hasVisualField {
				v := getTravelersAPIValue(fieldValue, visualField.Type)
				if v != nil {
					result[apiName] = v
				}
			}
		}
	}
	return result
}

func getTravelersAPIValue(fieldValue *travelersAPI.FieldValue, visualFieldType FieldType) interface{} {
	switch visualFieldType {
	case FieldTypeString:
		return fieldValue.GetStrValue()
	case FieldTypeBool:
		return fieldValue.GetBoolValue()
	case FieldTypeDate:
		v := fieldValue.GetDateValue()
		t := time.Date(int(v.Year), time.Month(v.Month), int(v.Day), 0, 0, 0, 0, time.UTC)
		return common.FormatDate(t)
	case FieldTypeGender:
		v := fieldValue.GetGenderValue()
		switch v {
		case travelersAPI.Gender_GENDER_MALE:
			return "male"
		case travelersAPI.Gender_GENDER_FEMALE:
			return "female"
		}
	case FieldTypeCountry:
		return fieldValue.GetCountryGeoIdValue()
	}
	return nil
}

func getTravelersAPIValueFromString(fieldValue string, visualFieldType FieldType) interface{} {
	switch visualFieldType {
	case FieldTypeString:
		return fieldValue
	case FieldTypeBool:
		if fieldValue == "true" {
			return true
		} else if fieldValue == "false" {
			return false
		}
	case FieldTypeDate:
		return fieldValue
	case FieldTypeGender:
		return fieldValue
	case FieldTypeCountry:
		v, err := strconv.ParseUint(fieldValue, 10, 64)
		if err == nil {
			return v
		}
	}
	return nil
}

func makeFieldsFromRaw(rawFields map[string]interface{}, visualEntity VisualEntity, visualFieldNameToTravelersAPIName map[string]string) map[string]*travelersAPI.FieldValue {
	fields := make(map[string]*travelersAPI.FieldValue)
	for visualFieldName, visualField := range visualEntity.VisualFieldDefinitions {
		attr, ok := visualFieldNameToTravelersAPIName[visualFieldName]
		if !ok {
			continue
		}
		rawValue, ok := rawFields[attr]
		if !ok || rawValue == nil {
			continue
		}
		switch visualField.Type {
		case FieldTypeString:
			switch vt := rawValue.(type) {
			case string:
				fields[visualFieldName] = &travelersAPI.FieldValue{Value: &travelersAPI.FieldValue_StrValue{StrValue: vt}}
			}
		case FieldTypeBool:
			switch vt := rawValue.(type) {
			case string:
				if vt == "true" {
					fields[visualFieldName] = &travelersAPI.FieldValue{Value: &travelersAPI.FieldValue_BoolValue{BoolValue: true}}
				} else if vt == "false" {
					fields[visualFieldName] = &travelersAPI.FieldValue{Value: &travelersAPI.FieldValue_BoolValue{BoolValue: false}}
				}
			case bool:
				fields[visualFieldName] = &travelersAPI.FieldValue{Value: &travelersAPI.FieldValue_BoolValue{BoolValue: vt}}
			}
		case FieldTypeCountry:
			switch vt := rawValue.(type) {
			case float64:
				v := uint64(vt)
				fields[visualFieldName] = &travelersAPI.FieldValue{Value: &travelersAPI.FieldValue_CountryGeoIdValue{CountryGeoIdValue: v}}
			}
		case FieldTypeDate:
			switch vt := rawValue.(type) {
			case string:
				d, _ := common.DateStringToProto(vt)
				if d != nil {
					fields[visualFieldName] = &travelersAPI.FieldValue{Value: &travelersAPI.FieldValue_DateValue{DateValue: d}}
				}
			}
		case FieldTypeGender:
			switch vt := rawValue.(type) {
			case string:
				if vt == "male" {
					fields[visualFieldName] = &travelersAPI.FieldValue{Value: &travelersAPI.FieldValue_GenderValue{GenderValue: travelersAPI.Gender_GENDER_MALE}}
				} else if vt == "female" {
					fields[visualFieldName] = &travelersAPI.FieldValue{Value: &travelersAPI.FieldValue_GenderValue{GenderValue: travelersAPI.Gender_GENDER_FEMALE}}
				}
			}
		}
	}
	return fields
}

func invertMap(input map[string]string) map[string]string {
	output := make(map[string]string, len(input))
	for k, v := range input {
		output[v] = k
	}
	return output
}

type Client interface {
	ListDocumentTypes(context.Context) (*travelersclient.DocumentTypes, error)
	GetTraveler(ctx context.Context, uid string) (*travelersclient.Traveler, error)
	CreateOrUpdateTraveler(ctx context.Context, uid string, traveler *travelersclient.EditableTraveler) (*travelersclient.Traveler, error)
	ListPassengers(ctx context.Context, uid string, includeCards bool, includeDocuments bool) ([]travelersclient.Passenger, error)
	GetPassenger(ctx context.Context, uid string, id string, includeCards bool, includeDocuments bool) (*travelersclient.Passenger, error)
	CreatePassenger(ctx context.Context, uid string, passenger *travelersclient.CreateOrUpdatePassengerRequest) (*travelersclient.Passenger, error)
	UpdatePassenger(ctx context.Context, uid string, id string, passenger *travelersclient.CreateOrUpdatePassengerRequest) (*travelersclient.Passenger, error)
	DeletePassenger(ctx context.Context, uid string, id string) error
	ListDocuments(ctx context.Context, uid, passengerID string) ([]travelersclient.Document, error)
	GetDocument(ctx context.Context, uid, passengerID, documentID string) (*travelersclient.Document, error)
	CreateDocument(ctx context.Context, uid, passengerID string, document *travelersclient.CreateOrUpdateDocumentRequest) (*travelersclient.Document, error)
	UpdateDocument(ctx context.Context, uid, passengerID, documentID string, document *travelersclient.CreateOrUpdateDocumentRequest) (*travelersclient.Document, error)
	DeleteDocument(ctx context.Context, uid, passengerID, documentID string) error
	ListBonusCards(ctx context.Context, uid, passengerID string) ([]travelersclient.BonusCard, error)
	GetBonusCard(ctx context.Context, uid, passengerID, bonusCardID string) (*travelersclient.BonusCard, error)
	CreateBonusCard(ctx context.Context, uid, passengerID string, bonusCard *travelersclient.EditableBonusCard) (*travelersclient.BonusCard, error)
	UpdateBonusCard(ctx context.Context, uid, passengerID, bonusCardID string, bonusCard *travelersclient.EditableBonusCard) (*travelersclient.BonusCard, error)
	DeleteBonusCard(ctx context.Context, uid, passengerID, bonusCardID string) error
}

type Cache interface {
	GetFieldsData() FieldsData
	RunUpdater()
}

type AdminCache interface {
	GetTag() string
	GetVisualPassenger() VisualEntity
	GetVisualDocumentsByType() map[string]VisualEntity
	GetVisualBonusCardsByType() map[string]VisualEntity
	GetTankerKeys() []string
}

type L10nService interface {
	Get(keysetName string, language string) (*l10n.Keyset, error)
}
