package travelers

import (
	"context"
	"sync/atomic"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelersclient"
	"a.yandex-team.ru/travel/app/backend/pkg/hashutils"
)

var KnownDocumentFields = []string{
	"title",
	"number",
	"first_name",
	"middle_name",
	"last_name",
	"first_name_en",
	"middle_name_en",
	"last_name_en",
	"issue_date",
	"expiration_date",
	"citizenship_geo_id",
}

type TravelersCacheConfig struct {
	UpdateInterval time.Duration `config:"TRAVELERS_CACHE_UPDATE_INTERVAL" yaml:"travelers_cache_update_interval"`
}

var DefaultTravelersCacheConfig = TravelersCacheConfig{
	UpdateInterval: 10 * time.Minute,
}

type FieldSettings struct {
	Regex    string
	Required bool
	Unused   bool
}

type DocumentType struct {
	Type           string
	FieldsSettings map[string]FieldSettings
}

type BonusCardType struct {
	Type           string
	FieldsSettings map[string]FieldSettings
}

type FieldsData struct {
	Tag             string
	PassengerFields map[string]FieldSettings
	DocumentTypes   []DocumentType
	BonusCardTypes  []BonusCardType
}

type TravelersCache struct {
	fieldsData      atomic.Value
	travelersClient *travelersclient.HTTPClient
	cfg             TravelersCacheConfig
	logger          log.Logger
}

var defaultPassengerFields = map[string]FieldSettings{
	"title": {
		Required: true,
		Unused:   false,
	},
	"gender": {
		Required: true,
		Unused:   false,
	},
	"birth_date": {
		Required: false,
		Unused:   false,
	},
	"phone": {
		Regex:    "^\\+?\\d{11,}$",
		Required: false,
		Unused:   false,
	},
	"phone_additional": {
		Regex:    "^\\+?\\d{11,}$",
		Required: false,
		Unused:   false,
	},
	"email": {
		Required: false,
		Unused:   false,
	},
	"itn": {
		Regex:    "^\\d{12}$",
		Required: false,
		Unused:   false,
	},
	"train_notifications_enabled": {
		Required: true,
		Unused:   false,
	},
}

var defaultBonusCardTypes = []BonusCardType{
	{
		Type: "rzd_bonus",
		FieldsSettings: map[string]FieldSettings{
			"number": {
				Regex:    "^\\d{13}$",
				Required: true,
				Unused:   false,
			},
			"title": {
				Required: false,
				Unused:   true,
			},
		},
	},
	{
		Type: "universal_road",
		FieldsSettings: map[string]FieldSettings{
			"number": {
				Regex:    "^\\d{13}$",
				Required: true,
				Unused:   false,
			},
			"title": {
				Required: false,
				Unused:   true,
			},
		},
	},
	{
		Type: "avia_company_loyalty",
		FieldsSettings: map[string]FieldSettings{
			"number": {
				Required: true,
				Unused:   false,
			},
			"title": {
				Required: false,
				Unused:   true,
			},
		},
	},
}

func NewTravelersCache(travelersClient *travelersclient.HTTPClient, cfg TravelersCacheConfig, logger log.Logger) (*TravelersCache, error) {
	tc := TravelersCache{
		travelersClient: travelersClient,
		cfg:             cfg,
		logger:          logger,
	}
	err := tc.update()
	if err != nil {
		return nil, err
	}
	return &tc, nil
}

func (c *TravelersCache) GetFieldsData() FieldsData {
	return c.fieldsData.Load().(FieldsData)
}

func (c *TravelersCache) RunUpdater() {
	t := time.NewTicker(c.cfg.UpdateInterval)
	defer t.Stop()
	for range t.C {
		err := c.update()
		// TODO (simon-ekb): send success/failure to solomon
		if err != nil {
			c.logger.Error("failed to update fields from travelers", log.Error(err))
			continue
		} else {
			c.logger.Info("successfully updated fields from travelers")
		}
	}
}

func (c *TravelersCache) update() error {
	tag, err := c.getNewTag()
	if err != nil {
		return err
	}

	currentData := c.fieldsData.Load()
	nothingToUpdate := currentData != nil && tag == currentData.(FieldsData).Tag
	if nothingToUpdate {
		return nil
	}

	dts, err := c.travelersClient.ListDocumentTypes(context.Background())
	if err != nil {
		return err
	}
	d := FieldsData{
		Tag:             tag,
		PassengerFields: defaultPassengerFields,
		DocumentTypes:   getDocumentTypesFromProto(dts),
		BonusCardTypes:  defaultBonusCardTypes,
	}
	c.logger.Debug("travelers cache", log.Any("data", d))
	c.fieldsData.Store(d)
	return nil
}

func (c *TravelersCache) getNewTag() (string, error) {
	passengerFieldsTag, err := hashutils.GetSimpleHashFromStruct(defaultPassengerFields)
	if err != nil {
		return "", err
	}
	bonusCardsTag, err := hashutils.GetSimpleHashFromStruct(defaultBonusCardTypes)
	if err != nil {
		return "", err
	}
	travelersVersionText, err := c.travelersClient.GetVersion(context.Background())
	if err != nil {
		return "", err
	}
	tag, err := hashutils.GetSimpleHash(passengerFieldsTag + travelersVersionText + bonusCardsTag)
	if err != nil {
		return "", err
	}
	return tag, nil
}

func getDocumentTypesFromProto(dts *travelersclient.DocumentTypes) []DocumentType {
	result := make([]DocumentType, 0, len(*dts))
	for dt, dtInfo := range *dts {
		requiredFields := make(map[string]struct{}, len(dtInfo.Required))
		for _, r := range dtInfo.Required {
			requiredFields[r] = struct{}{}
		}

		unusedFields := make(map[string]struct{}, len(dtInfo.Unused))
		for _, r := range dtInfo.Unused {
			unusedFields[r] = struct{}{}
		}

		fieldsSettings := make(map[string]FieldSettings)
		for _, field := range KnownDocumentFields {
			_, required := requiredFields[field]
			_, unused := unusedFields[field]
			fs := FieldSettings{
				Required: required,
				Unused:   unused,
			}
			regex, hasRegex := dtInfo.ReValidations[field]
			if hasRegex {
				fs.Regex = regex
			}
			fieldsSettings[field] = fs
		}

		r := DocumentType{
			Type:           dt,
			FieldsSettings: fieldsSettings,
		}
		result = append(result, r)
	}
	return result
}
