import logging
import os
import sys
from argparse import ArgumentParser
from datetime import datetime, timedelta

from travel.avia.ad_feed.ad_feed.click_price import ClickPriceCounter
from travel.avia.ad_feed.ad_feed.profit_cpa import ProfitCPA, Environment, CPA_ORDER_TABLE_FOR_ENV
from travel.avia.ad_feed.ad_feed.redir_balance_log import RedirBalanceLog, REDIR_BALANCE_LOG_DIR
from travel.avia.library.python.lib_yt.cache import Station
from travel.avia.library.python.lib_yt.client import configured_client
from yt.wrapper import TablePath

log = logging.getLogger(__name__)

OUTPUT_TABLE_FOR_ENVIRONMENT = {
    Environment.TESTING: '//home/avia/testing/data/ad-feed/click-price',
    Environment.PRODUCTION: '//home/avia/data/ad-feed/click-price',
}


def run(
    yt_proxy: str,
    yt_token: str,
    redir_balance_log_dir: str,
    records_since: datetime,
    national: str,
    cpa_orders_table: str,
    orders_since: datetime,
    output_table: str,
):
    yt_client = configured_client(yt_proxy, yt_token)
    redir_log = RedirBalanceLog(
        ytc=yt_client,
        stations=Station(yt_client),
        log_dir=redir_balance_log_dir,
        recrods_since=records_since,
        national_version=national,
    )
    profit_cpa = ProfitCPA(
        ytc=yt_client,
        orders_table=cpa_orders_table,
        orders_since=orders_since,
    )
    click_price = ClickPriceCounter(redir_log, profit_cpa).avg_click_price_by_direction

    yt_client.write_table(
        TablePath(
            output_table,
            schema=[
                {'name': 'from_id', 'type': 'int64'},
                {'name': 'to_id', 'type': 'int64'},
                {'name': 'price_cpc', 'type': 'double'},
                {'name': 'price_cpa', 'type': 'double'},
                {'name': 'clicks', 'type': 'uint64'},
                {'name': 'price_avg', 'type': 'double'},
            ],
        ),
        (
            {
                'from_id': direction.from_id,
                'to_id': direction.to_id,
                'price_cpc': price.price_cpc,
                'price_cpa': price.price_cpa,
                'clicks': price.clicks,
                'price_avg': price.price_avg,
            }
            for direction, price in click_price.items()
        ),
    )


def main():
    logging.basicConfig(
        level=logging.INFO,
        stream=sys.stdout,
    )

    parser = ArgumentParser()

    parser.add_argument('--yt-proxy', type=str, choices=('hahn', 'arnold'), default='hahn')
    parser.add_argument('--yt-token', type=str, default=os.getenv('YT_TOKEN'))
    parser.add_argument('--orders-age', type=int, default=30)
    parser.add_argument('--redir-age', type=int, default=30)
    parser.add_argument('--national-version', type=str, choices=('ru', 'ua', 'kz', 'com', 'tr'), required=True)

    subparser = parser.add_subparsers(
        title='mode',
        dest='mode',
        help='Automatically or manually fill table paths. '
        'Auto mode is convenient for usage in a particular environment. '
        'Manual mode is useful for debugging and development.',
        required=True,
    )

    parser_auto = subparser.add_parser('auto')
    parser_auto.add_argument('environment', type=Environment, choices=list(Environment))

    parser_manual = subparser.add_parser('manual')
    parser_manual.add_argument(
        '--redir-balance-log-dir', type=str, required=True, help='Directory with redir-balance-log-tables'
    )
    parser_manual.add_argument('--cpa-order-table', type=str, required=True)
    parser_manual.add_argument('--output-table', type=str, required=True)

    arguments = parser.parse_args()

    if 'auto' == arguments.mode:
        redir_balance_log_dir = REDIR_BALANCE_LOG_DIR
        cpa_orders_table = CPA_ORDER_TABLE_FOR_ENV[arguments.environment]
        output_table = OUTPUT_TABLE_FOR_ENVIRONMENT[arguments.environment]
    else:
        redir_balance_log_dir = arguments.redir_balance_log_dir
        cpa_orders_table = arguments.cpa_order_table
        output_table = arguments.output_table

    log.info(
        'Running with redir dir: %s, cpa_table: %s, output_table: %s',
        redir_balance_log_dir,
        cpa_orders_table,
        output_table,
    )

    now = datetime.now()
    run(
        arguments.yt_proxy,
        arguments.yt_token,
        redir_balance_log_dir,
        now - timedelta(days=arguments.redir_age),
        arguments.national_version,
        cpa_orders_table,
        now - timedelta(days=arguments.orders_age),
        output_table,
    )


if __name__ == '__main__':
    main()
