from functools import partial
from typing import Any, Callable
import datetime

from travel.avia.ad_feed.ad_feed.feed_generator.wizard_like import WizardLikeFeedRow
from travel.avia.ad_feed.ad_feed.converter.xml_converter import XmlYandexSmartBannersConverter, Category
from travel.avia.ad_feed.ad_feed.converter import CsvConverter


def identity(column: str, row: dict[str, Any]) -> Any:
    return row[column]


def new_name(oldcolumn: str, _: str, row: dict[str, Any]) -> Any:
    return row[oldcolumn]


def convert_currency(field: str) -> str:
    return field.replace('RUR', 'RUB')


_MapperType = dict[str, Callable[[str, Any], Any]]


def _ad_feed_column_mapper() -> _MapperType:
    column_mapper: _MapperType = {
        'departure_settlement_id*': partial(new_name, 'departure_settlement_id'),
        'arrival_settlement_id*': partial(new_name, 'arrival_settlement_id'),
        'departure_settlement_title*': partial(new_name, 'departure_settlement_title'),
        'arrival_settlement_title*': partial(new_name, 'arrival_settlement_title'),
        'url': partial(new_name, 'search_url'),
        'price': identity,
        'currency': identity,
        'route_url': lambda _, row: row['route_url'] or row['search_url'],
    }
    column_mapper.update(
        {
            name: identity
            for name in [
                'departure_settlement_title_from',
                'arrival_settlement_title_to',
                'forward_date',
                'backward_date',
                'search_url',
                'search_url_no_date',
                'direction_type',
                'departure_settlement_image',
                'arrival_settlement_image',
                'has_airport',
            ]
        },
    )
    column_mapper.update(
        {
            'sort1': partial(new_name, 'click_price'),
            'sort2': partial(new_name, 'flights_count'),
            'sort3': partial(new_name, 'popularity'),
        }
    )
    return column_mapper


def create_direction_feed_converter() -> CsvConverter:
    return CsvConverter(mapper=_ad_feed_column_mapper())


def create_stations_and_settlements_feed_converter() -> CsvConverter:
    column_mapper = _ad_feed_column_mapper()
    column_mapper.update(
        {
            'arrival_airport_title*': partial(new_name, 'arrival_airport_title'),
            'departure_airport_title*': partial(new_name, 'departure_airport_title'),
            'departure_airport_code*': partial(new_name, 'departure_airport_code'),
            'arrival_airport_code*': partial(new_name, 'arrival_airport_code'),
            'departure_titles_coincide': lambda _, row: (
                row['departure_settlement_title'] == row['departure_airport_title']
            ),
            'arrival_titles_coincide': lambda _, row: row['arrival_settlement_title'] == row['arrival_airport_title'],
        }
    )
    return CsvConverter(mapper=column_mapper)


def create_destination_only_feed_converter() -> CsvConverter:
    column_mapper: _MapperType = {
        'id': partial(new_name, 'arrival_settlement_id'),
        'url': partial(new_name, 'search_url'),
        'price': identity,
        'price_departure_msk': identity,
        'price_departure_spb': identity,
        'currency': identity,
    }
    column_mapper.update(
        {
            name: identity
            for name in [
                'arrival_settlement_title',
                'arrival_settlement_title_to',
                'forward_date',
                'backward_date',
                'search_url',
                'search_url_no_date',
                'direction_type',
                'arrival_settlement_image',
            ]
        },
    )
    column_mapper.update(
        {
            'sort1': partial(new_name, 'click_price'),
            'sort2': partial(new_name, 'flights_count'),
            'sort3': partial(new_name, 'popularity'),
        }
    )
    column_mapper['route_url'] = lambda _, row: row['route_url'] or row['search_url']
    return CsvConverter(mapper=column_mapper)  # type: ignore


def create_smart_banners_converter() -> CsvConverter:
    column_mapper: _MapperType = {
        'Origin ID': partial(new_name, 'departure_settlement_id'),
        'Origin name': partial(new_name, 'departure_settlement_title'),
        'Destination ID': partial(new_name, 'arrival_settlement_id'),
        'Destination name': partial(new_name, 'arrival_settlement_title'),
        'Final URL': partial(new_name, 'search_url'),
        'Flight price': lambda _, row: convert_currency(row['price_with_currency']),
        'Image URL': partial(new_name, 'arrival_settlement_image'),
    }
    return CsvConverter(mapper=column_mapper)  # type: ignore


def format_datetime(dt: datetime.datetime) -> str:
    return dt.strftime('%d.%m.%Y в %H:%M')


def make_name(row: WizardLikeFeedRow) -> str:
    return f"{row.from_settlement_title} - {row.to_settlement_title}, {format_datetime(row.departure_at)} - {format_datetime(row.arrival_at)}"


def make_description(row: WizardLikeFeedRow) -> str:
    return f"{row.from_settlement_title} - {row.to_settlement_title}, вылет {format_datetime(row.departure_at)}, прилёт {format_datetime(row.arrival_at)}"


def create_wizard_like_converter() -> XmlYandexSmartBannersConverter[WizardLikeFeedRow]:
    column_mapper: dict[str, Callable[[WizardLikeFeedRow], Any]] = {
        'name': make_name,
        'url': lambda row: row.url,
        'picture': lambda row: row.company_logo,
        'price': lambda row: row.price,
        'currencyId': lambda row: row.currency,
        'description': make_description,
        'categoryId': lambda row: row.direction_id,
    }
    return XmlYandexSmartBannersConverter[WizardLikeFeedRow](
        mapper=column_mapper,
        category_extractor=lambda row: Category(
            id=row.direction_id, name=f"{row.from_settlement_title} - {row.to_settlement_title}"
        ),
    )
