import logging
import os
import tempfile
from typing import Optional

from dataclasses import dataclass
from yt.wrapper import YtClient, TablePath

from travel.avia.ad_feed.ad_feed.validation import K50Validator
from travel.avia.ad_feed.ad_feed.converter import CsvConverter
from travel.avia.library.python.boto3_entities import S3ClientProto
from travel.avia.ad_feed.ad_feed.environment import Environment

logger = logging.getLogger(__name__)

S3_ENDPOINT_FOR_ENV = {
    Environment.TESTING: 'https://s3.mdst.yandex.net',
    Environment.PRODUCTION: 'https://s3.mds.yandex.net',
}


@dataclass
class S3Path:
    prefix: str
    key: str
    bucket: str

    @property
    def path(self) -> str:
        return os.path.join(self.prefix, self.key)


def dump_yt_to_s3_csv(
    yt_path: str,
    csv_path: S3Path,
    s3_client: S3ClientProto,
    converter: CsvConverter,
    yt_client: YtClient,
    validator: Optional[K50Validator],
) -> None:
    with tempfile.NamedTemporaryFile(mode='w+') as output_csv_file:
        logger.info('Dumping yt table to local temp csv file %s', output_csv_file.name)
        converter.convert(yt_client.read_table(TablePath(yt_path)), output_csv_file)
        logger.info('Done. File size is %d', os.path.getsize(output_csv_file.name))
        output_csv_file.flush()

        if validator is not None:
            output_csv_file.seek(0)
            validator.validate(output_csv_file)

        upload_to_s3(csv_path, output_csv_file.name, s3_client)


def upload_to_s3(remote_path: S3Path, local_file: str, s3_client: S3ClientProto) -> None:
    logger.info(
        'Uploading local file %s to S3 bucket %s with key %s', local_file, remote_path.bucket, remote_path.prefix
    )
    s3_client.upload_file(
        local_file,
        remote_path.bucket,
        remote_path.path,
    )
    logger.info(
        'Listing files in bucket %s with prefix %s\n%s',
        remote_path.bucket,
        remote_path.prefix,
        s3_client.list_objects(Bucket=remote_path.bucket, Prefix=remote_path.prefix)['Contents'],
    )
