import datetime
import xml.etree.cElementTree as ET
from typing import Callable, Any, Iterator, IO, Generic, TypeVar

from pydantic import BaseModel


class Category(BaseModel):
    id: int
    name: str


T = TypeVar('T')


class XmlYandexSmartBannersConverter(Generic[T]):
    def __init__(self, mapper: dict[str, Callable[[T], Any]], category_extractor: Callable[[T], Category]):
        self._mapper = mapper
        self._category_extractor = category_extractor

    def convert(self, input_data: Iterator[T], output: IO[Any]) -> None:
        root = ET.Element("avia-feed")
        ET.SubElement(root, "generation-date").text = datetime.datetime.now().strftime("%Y-%m-%dT%H:%M")

        categories_values = {}
        categories = ET.SubElement(root, "categories")
        offers = ET.SubElement(root, "offers")
        for i, row in enumerate(input_data):
            category = self._category_extractor(row)
            categories_values[category.id] = category.name

            offer = ET.SubElement(offers, "offer", id=str(i))
            for key, mapper in self._mapper.items():
                ET.SubElement(offer, key).text = str(mapper(row))

        for key, name in categories_values.items():  # type: ignore
            category = ET.SubElement(categories, "category", id=str(key))  # type: ignore
            category.text = name  # type: ignore
        output.write(ET.tostring(root, encoding='utf-8', xml_declaration=True))
