import logging
import os
from argparse import ArgumentParser

import sys
from pydantic import BaseModel, Field
from yt.wrapper import YtClient

from travel.avia.ad_feed.ad_feed.direction_flights import FlightsCounter
from travel.avia.ad_feed.ad_feed.direction_flights import OUTPUT_TABLE_FOR_ENVIRONMENT
from travel.avia.ad_feed.ad_feed.dumper import YtDumper
from travel.avia.ad_feed.ad_feed.environment import Environment
from travel.avia.library.python.lib_yt.cache import Station
from travel.avia.library.python.lib_yt.client import configured_client
from travel.avia.library.python.shared_flights_client.client import SharedFlightsClient

log = logging.getLogger(__name__)


class DirectionFlights(BaseModel):
    from_id: int = Field(yt_type='int64')
    to_id: int = Field(yt_type='int64')
    flights: int = Field(yt_type='int64')


def run(client: YtClient, output_table: str) -> None:
    counter = FlightsCounter(SharedFlightsClient(), Station(client))
    YtDumper(client).dump(
        (
            DirectionFlights(from_id=direction.from_id, to_id=direction.to_id, flights=count)
            for direction, count in counter.flights_by_direction.items()
        ),
        output_table=output_table,
    )


def main():
    logging.basicConfig(
        level=logging.INFO,
        stream=sys.stdout,
    )

    parser = ArgumentParser()

    parser.add_argument('--yt-proxy', type=str, choices=('hahn', 'arnold'), default='hahn')
    parser.add_argument('--yt-token', type=str, default=os.getenv('YT_TOKEN'))

    subparser = parser.add_subparsers(
        title='mode',
        dest='mode',
        help='Automatically or manually fill table paths. '
        'Auto mode is convenient for usage in a particular environment. '
        'Manual mode is useful for debugging and development.',
        required=True,
    )

    parser_auto = subparser.add_parser('auto')
    parser_auto.add_argument('environment', type=Environment, choices=list(Environment))

    parser_manual = subparser.add_parser('manual')
    parser_manual.add_argument('--output-table', type=str, required=True)

    arguments = parser.parse_args()

    if 'auto' == arguments.mode:
        output_table = OUTPUT_TABLE_FOR_ENVIRONMENT[arguments.environment]
    else:
        output_table = arguments.output_table

    log.info('Running with output_table: %s', output_table)

    run(
        configured_client(arguments.yt_proxy, arguments.yt_token),
        output_table,
    )


if __name__ == '__main__':
    main()
