import logging
from enum import Enum
from typing import Optional

from travel.avia.library.python.lib_yt.cache import Station, Settlement

log = logging.getLogger(__name__)


class DirectionType(Enum):
    RUSSIA = 'RUSSIA'
    CIS = 'CIS'
    INTERNATIONAL = 'INTERNATIONAL'

    def __str__(self) -> str:
        return self.value

    def __repr__(self) -> str:
        return self.value


_RUSSIA_ID = 225
_CIS_COUNTRY_IDS = {
    149: 'Беларусь',
    159: 'Казахстан',
    167: 'Азербайджан',
    168: 'Армения',
    171: 'Узбекистан',
    207: 'Киргизия',
    208: 'Молдова',
    209: 'Таджикистан',
}


class DirectionTypeResolver:
    def __init__(self, stations: Station, settlements: Settlement):
        self.stations = stations
        self.settlements = settlements

    def for_point(self, point_key: str) -> Optional[DirectionType]:
        if not point_key:
            return None
        return self._for_country(self._as_country(point_key))

    def _reduce_direction_types(self, *types: Optional[DirectionType]) -> Optional[DirectionType]:
        if not (types and all(types)):
            return None
        if all(d == DirectionType.RUSSIA for d in types):
            return DirectionType.RUSSIA
        elif all(d in (DirectionType.CIS, DirectionType.RUSSIA) for d in types):
            return DirectionType.CIS
        return DirectionType.INTERNATIONAL

    def for_settlement_pair(self, from_id: int, to_id: int) -> Optional[DirectionType]:
        d_from = self._for_country(self._settlement_to_country(from_id))
        d_to = self._for_country(self._settlement_to_country(to_id))
        return self._reduce_direction_types(d_from, d_to)

    def for_point_pair(self, from_code: str, to_code: str) -> Optional[DirectionType]:
        d_from, d_to = self.for_point(from_code), self.for_point(to_code)
        return self._reduce_direction_types(d_from, d_to)

    def for_settlement(self, to_id: int) -> Optional[DirectionType]:
        return self._for_country(self._settlement_to_country(to_id))

    def _for_country(self, country_id: Optional[int]) -> Optional[DirectionType]:
        if not country_id:
            return None
        if country_id == _RUSSIA_ID:
            return DirectionType.RUSSIA
        elif country_id in _CIS_COUNTRY_IDS:
            return DirectionType.CIS
        else:
            return DirectionType.INTERNATIONAL

    def _settlement_to_country(self, settlement_id: int) -> Optional[int]:
        settlement = self.settlements.by_id.get(settlement_id)
        if not settlement:
            log.warning('Unknown settlement: %s', settlement_id)
            return None
        country_code = settlement['country_id']
        if not country_code:
            log.warning('Settlement with no country: %s', settlement_id)
            return None
        return int(country_code[1:])

    def _station_to_country(self, station_id: int) -> Optional[int]:
        station = self.stations.by_id.get(station_id)
        if not station:
            log.warning('Unknown station: %s', station_id)
            return None
        country_code = station['country_id']
        if not country_code:
            log.warning('Station with no country: %s', station_id)
            return None
        return int(country_code[1:])

    def _as_country(self, point_key: str) -> Optional[int]:
        if not point_key:
            return None
        if point_key[0] == 'l':
            return int(point_key[1:])
        elif point_key[0] == 'c':
            return self._settlement_to_country(int(point_key[1:]))
        elif point_key[0] == 's':
            return self._station_to_country(int(point_key[1:]))
        log.warning('Unknown point key: %s', point_key)
        return None
