import logging
from typing import Iterator, Type
from urllib.parse import urlencode

from more_itertools import peekable
from pydantic import BaseModel
from yt.wrapper import YtClient, TablePath

logger = logging.getLogger(__name__)


def get_schema(model_type: Type[BaseModel]) -> list[dict[str, str]]:
    result = []
    for name, value in model_type.schema()['properties'].items():
        if 'yt_type' in value:
            result.append({'name': name, 'type': value['yt_type']})
            continue
        raise ValueError('Can not make a schema from a model of type %s', model_type)
    return result


class YtDumper:
    def __init__(self, client: YtClient):
        self._client = client

    def dump(self, data: Iterator[BaseModel], output_table: str) -> None:
        data = peekable(data)
        first = data.peek(None)
        if first is None:
            logger.warning("No data provided")
            return
        with self._client.Transaction():
            self._client.write_table(
                TablePath(output_table, schema=get_schema(type(first))),
                (row.dict() for row in data),
            )

            table_url = f'https://yt.yandex-team.ru/hahn/navigation?{urlencode(dict(path=output_table))}'
            logger.info('Dumped to %s %s', output_table, table_url)
