from typing import Final, TypeVar, Iterator, Type, Generic

import yt.clickhouse as chyt
from pydantic import BaseModel
from yt.wrapper import YtClient

from travel.avia.ad_feed.ad_feed.dumper import YtDumper
from travel.avia.ad_feed.ad_feed.feed_generator.direction import DirectionFeedRow

ROWS_MAX_NUMBER: Final = 10 ** 5


T = TypeVar('T', bound=BaseModel)

TFeedRow = TypeVar('TFeedRow', bound=DirectionFeedRow)


class ByClickPriceCutter(Generic[TFeedRow]):
    def __init__(self, threshold: float):
        self._threshold = threshold

    def __call__(self, stream: Iterator[TFeedRow]) -> Iterator[TFeedRow]:
        for value in stream:
            if value.click_price is None or value.click_price < self._threshold:
                continue
            yield value


class ByPopularityCutter(Generic[T]):
    def __init__(self, t: Type[T], number: int, yt_client: YtClient, intermediate_table: str):
        self._t = t
        self._number = number
        self._yt_client = yt_client
        self._intermediate_table = intermediate_table

    def __call__(self, stream: Iterator[T]) -> Iterator[T]:
        YtDumper(client=self._yt_client).dump(stream, output_table=self._intermediate_table)
        yield from map(
            self._t.parse_obj,
            chyt.execute(
                f"select * from `{self._intermediate_table}` order by popularity desc limit {self._number}",
                alias="*ch_public",
                client=self._yt_client,
            ),
        )
