import logging
from typing import Optional, Iterator

from pydantic import BaseModel, ValidationError, Field

from travel.avia.ad_feed.ad_feed.airport_blacklist import AirportBlacklist
from travel.avia.ad_feed.ad_feed.city_url import CityUrlGetter
from travel.avia.ad_feed.ad_feed.click_price import ClickPriceCounter
from travel.avia.ad_feed.ad_feed.direction_flights import FlightsCounter
from travel.avia.ad_feed.ad_feed.direction_type import DirectionTypeResolver
from travel.avia.ad_feed.ad_feed.entities import SettlementId
from travel.avia.ad_feed.ad_feed.environment import Environment
from travel.avia.ad_feed.ad_feed.feed_generator.abstract import IFeedGenerator
from travel.avia.ad_feed.ad_feed.feed_generator.const import DEFAULT_IMAGES
from travel.avia.ad_feed.ad_feed.min_price import MinPriceGetter
from travel.avia.ad_feed.ad_feed.top_directions import StationsPopularityGetter
from travel.avia.library.python.lib_yt.cache import SettlementBigImage
from travel.avia.library.python.urls import TravelAviaSearch, TravelAviaPrefilledFields

logger = logging.getLogger(__name__)

OUTPUT_TABLE_FOR_ENVIRONMENT = {
    Environment.TESTING: '//home/avia/testing/data/ad-feed/ru/destination-only-feed',
    Environment.PRODUCTION: '//home/avia/data/ad-feed/ru/destination-only-feed',
}

TRAVEL_HOST_BY_ENV = {
    Environment.TESTING: 'travel-test.yandex.ru',
    Environment.PRODUCTION: 'travel.yandex.ru',
}


class DestinationOnlyFeedRow(BaseModel):
    arrival_settlement_id: SettlementId = Field(yt_type='int64')
    arrival_settlement_title: str = Field(yt_type='string')
    arrival_settlement_title_to: str = Field(yt_type='string')
    forward_date: str = Field(yt_type='string')
    backward_date: Optional[str] = Field(yt_type='string')
    price: float = Field(yt_type='double')
    price_departure_msk: float = Field(yt_type='double')
    price_departure_spb: float = Field(yt_type='double')
    currency: str = Field(yt_type='string')
    search_url: str = Field(yt_type='string')
    search_url_no_date: str = Field(yt_type='string')
    click_price: Optional[float] = Field(yt_type='double')
    flights_count: int = Field(yt_type='uint64')
    direction_type: str = Field(yt_type='string')
    popularity: Optional[int] = Field(yt_type='uint64')
    arrival_settlement_image: str = Field(yt_type='string')
    route_url: Optional[str] = Field(yt_type='string')


class DestinationOnlyFeedGenerator(IFeedGenerator[DestinationOnlyFeedRow]):
    def __init__(
        self,
        min_prices: MinPriceGetter,
        click_price: ClickPriceCounter,
        direction_flights: FlightsCounter,
        direction_type: DirectionTypeResolver,
        stations_popularity_getter: StationsPopularityGetter,
        images: SettlementBigImage,
        search_url_builder: TravelAviaSearch,
        prefilled_url_builder: TravelAviaPrefilledFields,
        city_url_builder: CityUrlGetter,
        airport_blacklist: AirportBlacklist,
    ):
        self._min_prices = min_prices
        self.click_price = click_price
        self.direction_flights = direction_flights
        self.direction_type = direction_type
        self._stations_popularity_getter = stations_popularity_getter
        self.images = images
        self._search_url_builder = search_url_builder
        self._prefilled_url_builder = prefilled_url_builder
        self._city_url_builder = city_url_builder
        self._airport_blacklist = airport_blacklist

    def generate_feed(self) -> Iterator[DestinationOnlyFeedRow]:
        logger.info('Fetching average click price')
        click_prices = self.click_price.avg_click_price_by_settlement
        logger.info('Fetching flights count by settlement')
        station_flights = self.direction_flights.flights_by_arrival_settlement
        logger.info('Fetching top directions')
        stations_popularity = self._stations_popularity_getter.stations_popularity

        logger.info('Iterating through min prices')
        for min_price_row in self._min_prices.iterate_min_settlement_prices():
            to_id = SettlementId(int(min_price_row.arrival_settlement_id))

            if self._airport_blacklist.contains_settlement(to_id):
                logger.info(f'Skipping row by {to_id=}')
                continue

            try:
                yield DestinationOnlyFeedRow(
                    arrival_settlement_id=to_id,
                    arrival_settlement_title=min_price_row.arrival_settlement_title,
                    arrival_settlement_title_to=min_price_row.arrival_settlement_title_to,
                    forward_date=min_price_row.forward_date,
                    backward_date=min_price_row.backward_date,
                    price=min_price_row.price,
                    price_departure_msk=min_price_row.price_departure_msk,
                    price_departure_spb=min_price_row.price_departure_spb,
                    currency=min_price_row.currency,
                    search_url=self._prefilled_url_builder.url(
                        from_id="-me",
                        to_id=to_id,
                        when=min_price_row.forward_date,
                        return_date=min_price_row.backward_date,
                    ),
                    search_url_no_date=self._search_url_builder.url(
                        from_id="-me",
                        to_id=to_id,
                    ),
                    click_price=click_prices[to_id].price_avg if to_id in click_prices else None,
                    flights_count=station_flights.get(to_id),
                    direction_type=str(self.direction_type.for_settlement(to_id)),
                    popularity=stations_popularity.get(to_id),
                    arrival_settlement_image=self.images.get_marketing_square(
                        int(min_price_row.arrival_settlement_id),
                        DEFAULT_IMAGES,
                    ),
                    route_url=self._city_url_builder.url(city_id=to_id),
                )
            except ValidationError as e:
                logger.exception(e)
