import logging
import os
from datetime import datetime, timedelta
from functools import lru_cache, partial
from typing import Final, Any, Iterator, TypeVar

import jinja2
import yaml
from solomon import BasePushApiReporter
from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient

from travel.avia.ad_feed.ad_feed.city_url import CityUrlGetter
from travel.avia.ad_feed.ad_feed.feed_generator.wizard_like import (
    DIRECTIONS_TABLE,
    OrderUrlBuilder,
    WizardClient,
    WizardLikeFeedRow,
    WizardLikeFeedGenerator,
)
from library.python import resource
from travel.avia.ad_feed.ad_feed.airport_blacklist import AirportBlacklist
from travel.avia.ad_feed.ad_feed.click_price import ClickPriceCounter
from travel.avia.ad_feed.ad_feed.direction_flights import FlightsCounter
from travel.avia.ad_feed.ad_feed.direction_type import DirectionTypeResolver
from travel.avia.ad_feed.ad_feed.environment import Environment
from travel.avia.ad_feed.ad_feed.feed_generator.alias_adder import DirectionCityAliasAdder
from travel.avia.ad_feed.ad_feed.feed_generator.cutter import (
    ByPopularityCutter,
    ROWS_MAX_NUMBER as CUTTER_ROWS_NUMBER,
    ByClickPriceCutter,
)
from travel.avia.ad_feed.ad_feed.feed_generator.destination_only import (
    DestinationOnlyFeedGenerator,
    DestinationOnlyFeedRow,
)
from travel.avia.ad_feed.ad_feed.feed_generator.direction import DirectionFeedGenerator, DirectionFeedRow
from travel.avia.ad_feed.ad_feed.feed_generator.pipeline import Pipeline
from travel.avia.ad_feed.ad_feed.feed_generator.stations_and_settlements import (
    StationsAndSettlementsFeedGenerator,
    StationsAndSettlementsRow,
)
from travel.avia.ad_feed.ad_feed.feed_generator.yql import YqlFeedGenerator
from travel.avia.ad_feed.ad_feed.metrics import record_counter, create_solomon_reporter, SolomonSettings
from travel.avia.ad_feed.ad_feed.min_price import MinPriceGetter
from travel.avia.ad_feed.ad_feed.profit_cpa import ProfitCPA
from travel.avia.ad_feed.ad_feed.redir_balance_log import RedirBalanceLog
from travel.avia.ad_feed.ad_feed.supplier import CityAliasSupplier
from travel.avia.ad_feed.ad_feed.supplier.airport_info import AirportInfoSupplier
from travel.avia.ad_feed.ad_feed.top_directions import TopDirections, StationsPopularityGetter
from travel.avia.library.python.lib_yt.cache import Station, Settlement, SettlementBigImage
from travel.avia.library.python.shared_dicts.cache.settlement_cache import SettlementCache
from travel.avia.library.python.shared_flights_client.client import SharedFlightsClient
from travel.avia.library.python.urls import TravelAviaSearch, TravelAviaPrefilledFields

log = logging.getLogger(__name__)

SHARED_FLIGHTS_API_URL_FOR_ENV = {
    Environment.TESTING: 'http://shared-flights.testing.avia.yandex.net',
    Environment.PRODUCTION: 'http://shared-flights.production.avia.yandex.net',
}


@lru_cache(maxsize=None)
def get_solomon_reporter() -> BasePushApiReporter:
    return create_solomon_reporter(SolomonSettings())


T = TypeVar('T')


def count_records(stream: Iterator[T]) -> Iterator[T]:
    yield from record_counter(stream, reporter=get_solomon_reporter(), sensor='rows_number')


def create_direction_feed_generator(
    client: YtClient,
    min_price_table: str,
    redir_balance_log_dir: str,
    redir_balance_log_since: datetime,
    national_version: str,
    cpa_orders_table: str,
    cpa_orders_since: datetime,
    shared_flights_api_url: str,
    search_log_dir: str,
    search_log_since: datetime,
    airport_supplier: AirportInfoSupplier,
    city_alias_supplier: CityAliasSupplier,
    intermediate_table: str,
    click_price_threshold: float,
    airport_blacklist: AirportBlacklist,
) -> Pipeline[DirectionFeedRow]:
    log.info(
        'Running feed generator with: national_version=%s, min_price_table=%s, '
        'redir_balance_log_dir=%s, redir_balance_log_since=%s, '
        'cpa_orders_table=%s, cpa_orders_since=%s, '
        'shared_flights_api_url=%s, search_log_dir=%s, search_log_since=%s',
        national_version,
        min_price_table,
        redir_balance_log_dir,
        redir_balance_log_since,
        cpa_orders_table,
        cpa_orders_since,
        shared_flights_api_url,
        search_log_dir,
        search_log_since,
    )
    basic = _create_basic_directional_generator(
        airport_supplier=airport_supplier,
        client=client,
        cpa_orders_since=cpa_orders_since,
        cpa_orders_table=cpa_orders_table,
        min_price_table=min_price_table,
        national_version=national_version,
        redir_balance_log_dir=redir_balance_log_dir,
        redir_balance_log_since=redir_balance_log_since,
        search_log_dir=search_log_dir,
        search_log_since=search_log_since,
        shared_flights_api_url=shared_flights_api_url,
        airport_blacklist=airport_blacklist,
    )
    alias_adder = DirectionCityAliasAdder(city_alias_supplier=city_alias_supplier)
    price_cutter = ByClickPriceCutter[DirectionFeedRow](threshold=click_price_threshold)
    popularity_cutter = ByPopularityCutter[DirectionFeedRow](
        t=DirectionFeedRow,
        number=CUTTER_ROWS_NUMBER,
        yt_client=client,
        intermediate_table=intermediate_table,
    )
    return Pipeline[DirectionFeedRow](
        [lambda _: basic.generate_feed(), alias_adder, count_records, price_cutter, popularity_cutter]
    )


def _create_basic_directional_generator(
    client: YtClient,
    min_price_table: str,
    redir_balance_log_dir: str,
    redir_balance_log_since: datetime,
    national_version: str,
    cpa_orders_table: str,
    cpa_orders_since: datetime,
    shared_flights_api_url: str,
    search_log_dir: str,
    search_log_since: datetime,
    airport_supplier: AirportInfoSupplier,
    airport_blacklist: AirportBlacklist,
) -> DirectionFeedGenerator:
    yt_station = Station(client)
    yt_settlement = Settlement(client)
    yt_images = SettlementBigImage(client)
    return DirectionFeedGenerator(
        min_prices=MinPriceGetter(
            client=client,
            table=min_price_table,
        ),
        click_price=ClickPriceCounter(
            redir_balance_log=RedirBalanceLog(
                ytc=client,
                stations=yt_station,
                log_dir=redir_balance_log_dir,
                recrods_since=redir_balance_log_since,
                national_version=national_version,
            ),
            profit_cpa=ProfitCPA(
                ytc=client,
                orders_table=cpa_orders_table,
                orders_since=cpa_orders_since,
            ),
        ),
        direction_flights=FlightsCounter(
            shared_flights_client=SharedFlightsClient(
                api_uri=shared_flights_api_url,
            ),
            stations=yt_station,
        ),
        direction_type=DirectionTypeResolver(
            stations=yt_station,
            settlements=yt_settlement,
        ),
        top_directions=TopDirections(
            client=client,
            stations=yt_station,
            search_log_dir=search_log_dir,
            records_since=search_log_since,
            national_version=national_version,
        ),
        images=yt_images,
        airport_info_supplier=airport_supplier,
        settlement_info_supplier=SettlementCache(logger=log),
        airport_blacklist=airport_blacklist,
    )


def create_destination_only_generator(
    client: YtClient,
    min_price_table: str,
    redir_balance_log_dir: str,
    redir_balance_log_since: datetime,
    national_version: str,
    cpa_orders_table: str,
    cpa_orders_since: datetime,
    shared_flights_api_url: str,
    search_log_dir: str,
    search_log_since: datetime,
    travel_host: str,
    intermediate_table: str,
    city_url_builder: CityUrlGetter,
    airport_blacklist: AirportBlacklist,
) -> Pipeline[DestinationOnlyFeedRow]:
    log.info(
        'Create settlement ad feed generator with: national_version=%s, min_price_table=%s, '
        'redir_balance_log_dir=%s, redir_balance_log_since=%s, '
        'cpa_orders_table=%s, cpa_orders_since=%s, '
        'shared_flights_api_url=%s, search_log_dir=%s, search_log_since=%s',
        national_version,
        min_price_table,
        redir_balance_log_dir,
        redir_balance_log_since,
        cpa_orders_table,
        cpa_orders_since,
        shared_flights_api_url,
        search_log_dir,
        search_log_since,
    )
    yt_station = Station(client)
    yt_settlement = Settlement(client)
    yt_images = SettlementBigImage(client)
    main = DestinationOnlyFeedGenerator(
        min_prices=MinPriceGetter(
            client=client,
            table=min_price_table,
        ),
        click_price=ClickPriceCounter(
            redir_balance_log=RedirBalanceLog(
                ytc=client,
                stations=yt_station,
                log_dir=redir_balance_log_dir,
                recrods_since=redir_balance_log_since,
                national_version=national_version,
            ),
            profit_cpa=ProfitCPA(
                ytc=client,
                orders_table=cpa_orders_table,
                orders_since=cpa_orders_since,
            ),
        ),
        direction_flights=FlightsCounter(
            shared_flights_client=SharedFlightsClient(
                api_uri=shared_flights_api_url,
            ),
            stations=yt_station,
        ),
        direction_type=DirectionTypeResolver(
            stations=yt_station,
            settlements=yt_settlement,
        ),
        stations_popularity_getter=StationsPopularityGetter(
            client=client,
            stations=yt_station,
            search_log_dir=search_log_dir,
            records_since=search_log_since,
            national_version=national_version,
        ),
        images=yt_images,
        search_url_builder=TravelAviaSearch(travel_host=travel_host),
        prefilled_url_builder=TravelAviaPrefilledFields(travel_host=travel_host),
        city_url_builder=city_url_builder,
        airport_blacklist=airport_blacklist,
    )
    cutter = ByPopularityCutter[DestinationOnlyFeedRow](
        t=DestinationOnlyFeedRow,
        number=CUTTER_ROWS_NUMBER,
        yt_client=client,
        intermediate_table=intermediate_table,
    )
    return Pipeline[DestinationOnlyFeedRow]([lambda _: main.generate_feed(), count_records, cutter])


_FEED_CONFIG: Final = '/feeds/feeds.yaml'


def get_feed_config() -> dict[str, Any]:
    return yaml.safe_load(resource.find(_FEED_CONFIG))


def create_yql_generator(
    report_name: str,
    source_table: str,
    yt_client: YtClient,
    yql_client: YqlClient,
    output_table: str,
    airport_blacklist: AirportBlacklist,
) -> Pipeline[dict[str, Any]]:
    config = get_feed_config()[report_name]
    query_template = resource.find(os.path.join('/feeds/', config['template'])).decode('utf-8')
    query = jinja2.Template(query_template, undefined=jinja2.StrictUndefined).render(data=config)
    main = YqlFeedGenerator(
        query=query,
        source_table=source_table,
        output_table=output_table,
        yt_client=yt_client,
        yql_client=yql_client,
        airport_blacklist=airport_blacklist,
    )
    return Pipeline[dict[str, Any]](
        [
            lambda _: main.generate_feed(),
            partial(
                record_counter,
                reporter=get_solomon_reporter(),
                sensor='rows_number',
                labels={'smart_banners_report_name': report_name},
            ),
        ]
    )


def create_stations_and_settlements_generator(
    client: YtClient,
    min_price_table: str,
    redir_balance_log_dir: str,
    redir_balance_log_since: datetime,
    national_version: str,
    cpa_orders_table: str,
    cpa_orders_since: datetime,
    shared_flights_api_url: str,
    search_log_dir: str,
    search_log_since: datetime,
    stations_table: str,
    stations_mapping_table: str,
    intermediate_table: str,
    click_price_threshold: float,
    airport_blacklist: AirportBlacklist,
) -> Pipeline[StationsAndSettlementsRow]:
    airport_info_supplier = AirportInfoSupplier(
        yt_client=client, stations_table=stations_table, station2settlement_table=stations_mapping_table
    )
    directional_generator = _create_basic_directional_generator(
        client=client,
        min_price_table=min_price_table,
        redir_balance_log_dir=redir_balance_log_dir,
        redir_balance_log_since=redir_balance_log_since,
        national_version=national_version,
        cpa_orders_table=cpa_orders_table,
        cpa_orders_since=cpa_orders_since,
        shared_flights_api_url=shared_flights_api_url,
        search_log_dir=search_log_dir,
        search_log_since=search_log_since,
        airport_supplier=airport_info_supplier,
        airport_blacklist=airport_blacklist,
    )
    main = StationsAndSettlementsFeedGenerator(
        directional_generator=directional_generator,
        airport_info_supplier=airport_info_supplier,
        airport_blacklist=airport_blacklist,
    )
    cutter = ByPopularityCutter[StationsAndSettlementsRow](
        t=StationsAndSettlementsRow,
        number=CUTTER_ROWS_NUMBER,
        yt_client=client,
        intermediate_table=intermediate_table,
    )
    price_cutter = ByClickPriceCutter[StationsAndSettlementsRow](threshold=click_price_threshold)
    return Pipeline[StationsAndSettlementsRow]([lambda _: main.generate_feed(), count_records, cutter, price_cutter])


def create_wizard_like_generator(
    request_delay: timedelta,
    url_builder: OrderUrlBuilder,
    wizard_client: WizardClient,
    yt_client: YtClient,
    airport_blacklist: AirportBlacklist,
) -> Pipeline[WizardLikeFeedRow]:
    main = WizardLikeFeedGenerator(
        yt_client=yt_client,
        wizard_client=wizard_client,
        directions_table=DIRECTIONS_TABLE,
        variants_to_show=5,
        url_builder=url_builder,
        request_delay=request_delay,
        airport_blacklist=airport_blacklist,
    )
    return Pipeline[WizardLikeFeedRow]([lambda _: main.generate_feed(), count_records])
