import datetime
import logging
import time
from typing import Iterator, Sequence
from urllib.parse import urljoin
import hashlib

import requests
import urllib
from pydantic import BaseModel, Field
from yt.wrapper import YtClient, TablePath
from urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter

from travel.avia.ad_feed.ad_feed.airport_blacklist import AirportBlacklist
from travel.avia.ad_feed.ad_feed.feed_generator.const import DEFAULT_IMAGES
from travel.avia.ad_feed.ad_feed.feed_generator.abstract import IFeedGenerator
from travel.avia.ad_feed.ad_feed.supplier.airport_info import convert_city_id

logger = logging.getLogger(__name__)

DIRECTIONS_TABLE = '//home/avia/avia-statistics/landing-routes'


class Point(BaseModel):
    point_key: str
    id: int
    title: str


class Price(BaseModel):
    currency: str
    value: float = Field(alias='price')


class Settlement(BaseModel):
    code: str
    title: str


class Station(BaseModel):
    code: str
    title: str


class PointInfo(BaseModel):
    settlement: Settlement
    station: Station


class Flight(BaseModel):
    arrival: PointInfo
    departure: PointInfo
    number: str
    company_id: int
    arrives_at: datetime.datetime
    departs_at: datetime.datetime


class Route(BaseModel):
    duration: int
    flights: list[Flight]


class Variant(BaseModel):
    tariff: Price
    popularity: int
    forward: Route
    backward: Route


class Company(BaseModel):
    id: int
    title: str
    logo: str


class WizardResponse(BaseModel):
    to_point: Point = Field(alias='to')
    from_point: Point = Field(alias='from')
    variants: list[Variant]
    companies: list[Company]


def get_session() -> requests.Session:
    s = requests.Session()
    retries = Retry(total=5, backoff_factor=0.4, status_forcelist=[104, 413, 429, 503])
    s.mount('http://', HTTPAdapter(max_retries=retries))
    s.mount('https://', HTTPAdapter(max_retries=retries))
    return s


class WizardSettings(BaseModel):
    url: str
    variants_path: str

    @property
    def variants_url(self) -> str:
        return urljoin(self.url, self.variants_path)


class WizardClient:
    def __init__(self, settings: WizardSettings):
        self._session = get_session()
        self._settings = settings

    def get_variants(self, from_id: str, to_id: str) -> WizardResponse:
        response = self._session.get(
            self._settings.variants_url, params={'from_id': from_id, 'to_id': to_id, 'direct_flight': 1, 'lang': 'ru'}
        )

        response.raise_for_status()
        return WizardResponse.parse_obj(response.json())


class OrderUrlBuilder:
    def __init__(self, host: str):
        self._host = host

    def build(self, from_id: str, to_id: str, flights: Sequence[Flight]) -> str:
        encode_flights = ','.join(
            f'{flight.number}.{flight.departs_at.strftime("%Y-%m-%dT%H:%M")}' for flight in flights
        )
        params = urllib.parse.urlencode(
            {
                'adult_seats': 1,
                'backward': '',
                'children_seats': 0,
                'forward': encode_flights,
                'fromId': from_id,
                'infant_seats': 0,
                'klass': 'economy',
                'oneway': 1,
                'return_date': '',
                'toId': to_id,
                'when': str(flights[0].departs_at.date()),
            }
        )
        return f'https://{self._host}/avia/order/?{params}'


class WizardLikeFeedRow(BaseModel):
    url: str
    currency: str
    price: float
    from_settlement_title: str
    to_settlement_title: str
    company_logo: str
    departure_at: datetime.datetime
    arrival_at: datetime.datetime
    direction_id: int


def persistent_string_hash(s: str) -> int:
    return int(hashlib.md5(s.encode(encoding='utf-8')).hexdigest(), 16) % (10 ** 9 + 7)


def get_direction_id(from_id: str, to_id: str) -> int:
    return persistent_string_hash(from_id + '_' + to_id)


class WizardLikeFeedGenerator(IFeedGenerator[WizardLikeFeedRow]):
    def __init__(
        self,
        wizard_client: WizardClient,
        yt_client: YtClient,
        url_builder: OrderUrlBuilder,
        directions_table: str,
        variants_to_show: int,
        request_delay: datetime.timedelta,
        airport_blacklist: AirportBlacklist,
    ):
        self._wizard_client = wizard_client
        self._yt_client = yt_client
        self._url_builder = url_builder
        self._directions_table = directions_table
        self._variants_to_show = variants_to_show
        self._request_delay = request_delay
        self._airport_blacklist = airport_blacklist

    def generate_feed(self) -> Iterator[WizardLikeFeedRow]:
        for i, row in enumerate(
            self._yt_client.read_table(TablePath(self._directions_table, columns=('from_id', 'to_id')))
        ):
            from_id = f"c{row['from_id']}"
            to_id = f"c{row['to_id']}"

            if self._airport_blacklist.contains_settlement(convert_city_id(from_id)):
                logger.info(f'Skipping row by {from_id=}')
                continue
            if self._airport_blacklist.contains_settlement(convert_city_id(to_id)):
                logger.info(f'Skipping row by {to_id=}')
                continue

            try:
                response = self._wizard_client.get_variants(from_id=from_id, to_id=to_id)
            except requests.exceptions.HTTPError as e:
                logger.exception(e)
                continue
            if i % 1000 == 0:
                logger.info('Processed %s directions', i)
            company_by_id = {company.id: company for company in response.companies}
            for variant in response.variants[: self._variants_to_show]:
                company_logo = company_by_id[variant.forward.flights[0].company_id].logo
                if len(set(flight.company_id for flight in variant.forward.flights)) > 1:
                    company_logo = next(iter(DEFAULT_IMAGES))
                yield WizardLikeFeedRow(
                    currency=variant.tariff.currency,
                    price=variant.tariff.value,
                    from_settlement_title=response.from_point.title,
                    to_settlement_title=response.to_point.title,
                    url=self._url_builder.build(from_id=from_id, to_id=to_id, flights=variant.forward.flights),
                    company_logo=company_logo,
                    departure_at=variant.forward.flights[0].departs_at,
                    arrival_at=variant.forward.flights[-1].arrives_at,
                    direction_id=get_direction_id(from_id, to_id),
                )
            time.sleep(self._request_delay.total_seconds())
