import logging
from typing import Any, Iterator

from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder
from yt.wrapper import YtClient

from travel.avia.ad_feed.ad_feed.airport_blacklist import AirportBlacklist
from travel.avia.ad_feed.ad_feed.entities import SettlementId
from travel.avia.ad_feed.ad_feed.feed_generator.abstract import IFeedGenerator, GenerationError


logger = logging.getLogger(__name__)


class YqlFeedGenerator(IFeedGenerator[dict[str, Any]]):
    def __init__(
        self,
        query: str,
        source_table: str,
        output_table: str,
        yt_client: YtClient,
        yql_client: YqlClient,
        airport_blacklist: AirportBlacklist,
    ):
        self._query = query
        self._source_table = source_table
        self._output_table = output_table
        self._yt_client = yt_client
        self._yql_client = yql_client
        self._airport_blacklist = airport_blacklist

    def generate_feed(self) -> Iterator[dict[str, Any]]:
        with self._yt_client.Transaction(timeout=60000 * 5) as transaction:
            result_table = self._yt_client.create_temp_table()
            query = self._yql_client.query(
                self._query,
                syntax_version=1,
                title='YQL: Generating smart banners feed',
            )
            result = query.run(
                transaction_id=transaction.transaction_id,
                parameters=ValueBuilder.build_json_map(
                    {
                        '$result_table': ValueBuilder.make_string(result_table),
                        '$source_table': ValueBuilder.make_string(self._source_table),
                    }
                ),
            )
            result.wait_progress()
            if not result.is_ok:
                raise GenerationError("Failed to generate feed")
            self._yt_client.move(result_table, self._output_table, force=True)
            for row in self._yt_client.read_table(self._yt_client.TablePath(self._output_table)):
                departure_settlement_id = row['departure_settlement_id']
                arrival_settlement_id = row['arrival_settlement_id']

                if self._airport_blacklist.contains_settlement(SettlementId(departure_settlement_id)):
                    logger.info(f'Skipping row by {departure_settlement_id=}')
                    continue
                if self._airport_blacklist.contains_settlement(SettlementId(arrival_settlement_id)):
                    logger.info(f'Skipping row by {arrival_settlement_id=}')
                    continue

                yield row
