from typing import Literal, Iterator, TypeVar, Optional
import logging
import datetime

from requests import HTTPError
from pydantic import BaseSettings
from solomon import OAuthProvider, BasePushApiReporter

from travel.avia.library.python.boto3_entities import S3ClientProto

from travel.avia.ad_feed.ad_feed.converter.runner import S3Path
from travel.avia.ad_feed.ad_feed.entities import S3ObjectInfo
from travel.avia.ad_feed.ad_feed.environment import Environment

logger = logging.getLogger(__name__)


class SolomonSettings(BaseSettings):
    project: Literal['avia'] = 'avia'
    cluster: str = 'ad_feed'
    url: str = 'http://solomon.yandex.net'
    token: str
    env: Environment = Environment.TESTING
    feed_name: str
    service: str = 'app'

    class Config:
        env_prefix = 'SOLOMON_'


def create_solomon_reporter(settings: SolomonSettings) -> BasePushApiReporter:
    auth_provider = OAuthProvider(settings.token)
    return BasePushApiReporter(
        project=settings.project,
        cluster=settings.cluster,
        service=settings.service,
        url=settings.url,
        common_labels={'feed_name': settings.feed_name, 'environment': settings.env.value},
        auth_provider=auth_provider,
    )


T = TypeVar('T')


def record_counter(
    stream: Iterator[T], reporter: BasePushApiReporter, sensor: str, labels: Optional[dict[str, str]] = None
) -> Iterator[T]:
    cnt = 0
    for e in stream:
        cnt += 1
        yield e
    if labels is None:
        labels = {}
    try:
        reporter.set_value(sensor=sensor, labels=labels, value=cnt)
    except HTTPError:
        logger.error("Can not send metrics", exc_info=True)


def _get_file_info(file_path: S3Path, client: S3ClientProto) -> datetime.datetime:
    response = S3ObjectInfo.parse_obj(client.get_object(Bucket=file_path.bucket, Key=file_path.path))
    return response.last_modified


def send_file_metrics(
    path: S3Path, client: S3ClientProto, reporter: BasePushApiReporter, labels: Optional[dict[str, str]] = None
) -> None:
    now_utc = datetime.datetime.now(datetime.timezone.utc)
    last_update = _get_file_info(path, client)
    delta = (now_utc - last_update).total_seconds()
    if labels is None:
        labels = {}
    try:
        reporter.set_value(sensor='update_delta', labels=labels, value=delta)
    except HTTPError:
        logger.error("Can not send metrics", exc_info=True)
