import logging
from datetime import datetime
from typing import Dict, Optional

from travel.avia.ad_feed.ad_feed.environment import Environment
from yt.wrapper import YtClient, TablePath

log = logging.getLogger(__name__)

CPA_ORDER_TABLE_FOR_ENV = {
    Environment.TESTING: '//home/travel/testing/cpa/avia/orders',
    Environment.PRODUCTION: '//home/travel/prod/cpa/avia/orders',
}


class ProfitCPA:
    def __init__(self, ytc: YtClient, orders_table: str, orders_since: datetime):
        self.client: YtClient = ytc
        self.orders_table = orders_table
        self.orders_age = orders_since
        self._cache: Optional[Dict[str, float]] = None

        log.info('ProfitCPA.__init__(orders_table=%s, orders_since=%s)', orders_table, orders_since)

    def get_profit(self) -> Dict[str, float]:
        """
        :return: A mapping of label(marker) to profit amount in rubbles
        """
        if not self._cache:
            self._cache = self._get_profit()
            log.info('Populated CPA cache with %d items', len(self._cache))
        return self._cache

    def _get_profit(self) -> Dict[str, float]:
        return {
            row['label']: float(row['profit_amount_ex_tax'])
            for row in self.client.read_table(
                TablePath(
                    self.orders_table,
                    columns=('label', 'profit_amount_ex_tax', 'created_at'),
                )
            )
            if datetime.fromtimestamp(row['created_at']) > self.orders_age
        }
