import datetime
import logging
from functools import lru_cache
from typing import Optional

import boto3
import click
from pydantic import BaseSettings
from yt.wrapper import YtClient

from travel.avia.ad_feed.ad_feed.airport_blacklist import AirportBlacklist
from travel.avia.ad_feed.ad_feed.city_url import CityUrlGetter, create_by_yt_tables
from travel.avia.ad_feed.ad_feed.metrics import send_file_metrics
from travel.avia.ad_feed.ad_feed.runner.tools import EnumType
from travel.avia.ad_feed.ad_feed.converter.factory import create_destination_only_feed_converter
from travel.avia.ad_feed.ad_feed.converter.runner import S3Path, dump_yt_to_s3_csv
from travel.avia.ad_feed.ad_feed.dumper import YtDumper
from travel.avia.ad_feed.ad_feed.entities import NationalVersion
from travel.avia.library.python.boto3_entities import S3ClientProto
from travel.avia.ad_feed.ad_feed.environment import Environment
from travel.avia.ad_feed.ad_feed.feed_generator.destination_only import (
    OUTPUT_TABLE_FOR_ENVIRONMENT,
)
from travel.avia.ad_feed.ad_feed.feed_generator.destination_only import TRAVEL_HOST_BY_ENV
from travel.avia.ad_feed.ad_feed.feed_generator.factory import (
    create_destination_only_generator,
    SHARED_FLIGHTS_API_URL_FOR_ENV,
    get_solomon_reporter,
)
from travel.avia.ad_feed.ad_feed.min_price import MIN_PRICE_TABLE_FOR_ENVIRONMENT
from travel.avia.ad_feed.ad_feed.profit_cpa import CPA_ORDER_TABLE_FOR_ENV
from travel.avia.ad_feed.ad_feed.redir_balance_log import REDIR_BALANCE_LOG_DIR
from travel.avia.ad_feed.ad_feed.settings import MdsSettings, YtSettings
from travel.avia.ad_feed.ad_feed.top_directions import AVIA_USER_SEARCH_LOG_DIR
from travel.avia.ad_feed.ad_feed.validation import K50Validator
from travel.avia.ad_feed.ad_feed.validation.k50 import ROWS_MAX_NUMBER
from travel.avia.library.python.lib_yt.client import configured_client

logger = logging.getLogger(__name__)


class AppSettings(BaseSettings):
    redir_balance_log_age: int = 90
    cpa_orders_age: int = 90
    search_log_age: int = 90
    redir_balance_log_dir: str = REDIR_BALANCE_LOG_DIR
    search_log_dir: str = AVIA_USER_SEARCH_LOG_DIR
    stations_table: str = '//home/rasp/reference/station'
    stations_mapping_table = '//home/rasp/reference/station2settlement'
    settlements_table: str = '//home/travel/prod/rasp_dicts/latest/settlement'
    landings_table: str = '//home/avia/avia-statistics/city-to-landing-cities'
    airport_blacklist_table: str = '//home/avia/data/ad-feed/blacklist'


@lru_cache(maxsize=None)
def create_validator() -> K50Validator:
    return K50Validator(rows_max_number=ROWS_MAX_NUMBER)


@lru_cache(maxsize=None)
def create_yt_client() -> YtClient:
    yt_settings = YtSettings()
    return configured_client(yt_settings.proxy, yt_settings.token)


@lru_cache(maxsize=None)
def create_s3_client() -> S3ClientProto:
    mds_settings = MdsSettings()
    return boto3.session.Session(
        aws_access_key_id=mds_settings.access_key_id,
        aws_secret_access_key=mds_settings.access_key_secret,
    ).client(
        service_name='s3',
        endpoint_url=mds_settings.endpoint,
    )


def create_city_url_getter(travel_host: str, yt_client: YtClient, app_settings: AppSettings) -> CityUrlGetter:
    return create_by_yt_tables(
        travel_host=travel_host,
        has_landing_table=app_settings.landings_table,
        slug_list_table=app_settings.settlements_table,
        yt_client=yt_client,
    )


def create_airport_blacklist(yt_client: YtClient, app_settings: AppSettings) -> AirportBlacklist:
    return AirportBlacklist(
        yt_client=yt_client,
        blacklist_table=app_settings.airport_blacklist_table,
        stations_table=app_settings.stations_table,
    )


def set_by_env(
    ctx: click.core.Context, _: click.core.Parameter, environment: Optional[Environment]
) -> Optional[Environment]:
    if environment is None:
        return None
    ctx.default_map = {
        'cpa_orders_table': CPA_ORDER_TABLE_FOR_ENV[environment],
        'shared_flights_api_url': SHARED_FLIGHTS_API_URL_FOR_ENV[environment],
        'output_table': OUTPUT_TABLE_FOR_ENVIRONMENT[environment],
    }
    return environment


@click.command()
@click.option(
    '--env',
    'environment',
    type=EnumType(Environment, case_sensitive=False),
    required=False,
    is_eager=True,
    callback=set_by_env,
)
@click.option('--national-version', type=click.Choice(list(NationalVersion), case_sensitive=False), required=True)
@click.option('--output-table', type=str, required=True)
@click.option('--min-price-table', type=str, required=False)
@click.option('--cpa-orders-table', type=str, required=True)
@click.option('--shared-flights-api-url', type=str, required=True)
@click.option('--mds-s3-bucket-name', type=str, default='avia-indexer', required=True)
@click.option('--mds-s3-prefix', type=str, default='destination-only-ad-feed', required=True)
@click.option('--mds-s3-key', type=str, default='ru.csv', required=False)
def main(
    national_version: NationalVersion,
    environment: Environment,
    output_table: str,
    min_price_table: str,
    shared_flights_api_url: str,
    cpa_orders_table: str,
    mds_s3_bucket_name: str,
    mds_s3_prefix: str,
    mds_s3_key: Optional[str],
):
    if environment is not None:
        min_price_table = MIN_PRICE_TABLE_FOR_ENVIRONMENT[environment].format(nv=national_version)
    if min_price_table is None:
        raise ValueError()

    app_settings = AppSettings()

    logger.info(
        'Running dumper with min_price_table %s, redir_balance_log_dir %s, '
        'cpa_orders_table %s, shared_flights_api_url %s, search_lo`g_dir %s, '
        'output_table %s',
        min_price_table,
        app_settings.redir_balance_log_dir,
        cpa_orders_table,
        shared_flights_api_url,
        app_settings.search_log_dir,
        output_table,
    )

    now = datetime.datetime.now()
    yt_client = create_yt_client()
    city_url_getter = create_city_url_getter(
        travel_host=TRAVEL_HOST_BY_ENV[environment], yt_client=yt_client, app_settings=app_settings
    )
    airport_blacklist = create_airport_blacklist(yt_client, app_settings)
    generator = create_destination_only_generator(
        client=yt_client,
        min_price_table=min_price_table,
        redir_balance_log_dir=app_settings.redir_balance_log_dir,
        redir_balance_log_since=now - datetime.timedelta(app_settings.redir_balance_log_age),
        national_version=national_version,
        cpa_orders_table=cpa_orders_table,
        cpa_orders_since=now - datetime.timedelta(app_settings.cpa_orders_age),
        shared_flights_api_url=shared_flights_api_url,
        search_log_dir=app_settings.search_log_dir,
        search_log_since=now - datetime.timedelta(app_settings.search_log_age),
        travel_host=TRAVEL_HOST_BY_ENV[environment],
        intermediate_table=f'{output_table}-full',
        city_url_builder=city_url_getter,
        airport_blacklist=airport_blacklist,
    )
    YtDumper(client=yt_client).dump(generator.generate_feed(), output_table=output_table)
    s3_client = create_s3_client()
    s3_path = S3Path(key=mds_s3_key, prefix=mds_s3_prefix, bucket=mds_s3_bucket_name)
    dump_yt_to_s3_csv(
        yt_path=output_table,
        csv_path=s3_path,
        s3_client=s3_client,
        yt_client=yt_client,
        converter=create_destination_only_feed_converter(),
        validator=create_validator(),
    )
    send_file_metrics(path=s3_path, client=s3_client, reporter=get_solomon_reporter())


if __name__ == '__main__':
    main()
