import logging
from datetime import datetime, timedelta
from functools import lru_cache
from typing import Optional

import boto3
import click
import sys
from pydantic import BaseSettings
from yt.wrapper import YtClient

from travel.avia.ad_feed.ad_feed.runner.tools import EnumType
from travel.avia.ad_feed.ad_feed.converter.factory import (
    create_direction_feed_converter,
    create_stations_and_settlements_feed_converter,
)
from travel.avia.ad_feed.ad_feed.converter.runner import S3Path, dump_yt_to_s3_csv
from travel.avia.ad_feed.ad_feed.dumper import YtDumper
from travel.avia.ad_feed.ad_feed.entities import NationalVersion
from travel.avia.library.python.boto3_entities import S3ClientProto
from travel.avia.ad_feed.ad_feed.environment import Environment
from travel.avia.ad_feed.ad_feed.feed_generator.direction import (
    OUTPUT_TABLE_FOR_ENVIRONMENT,
)
from travel.avia.ad_feed.ad_feed.feed_generator.factory import (
    create_direction_feed_generator,
    SHARED_FLIGHTS_API_URL_FOR_ENV,
    create_stations_and_settlements_generator,
    get_solomon_reporter,
)
from travel.avia.ad_feed.ad_feed.airport_blacklist import AirportBlacklist
from travel.avia.ad_feed.ad_feed.metrics import send_file_metrics
from travel.avia.ad_feed.ad_feed.min_price import MIN_PRICE_TABLE_FOR_ENVIRONMENT
from travel.avia.ad_feed.ad_feed.profit_cpa import CPA_ORDER_TABLE_FOR_ENV
from travel.avia.ad_feed.ad_feed.redir_balance_log import REDIR_BALANCE_LOG_DIR
from travel.avia.ad_feed.ad_feed.settings import MdsSettings, YtSettings, CityAliasSettings
from travel.avia.ad_feed.ad_feed.supplier import AirportInfoSupplier, CityAliasSupplier
from travel.avia.ad_feed.ad_feed.top_directions import AVIA_USER_SEARCH_LOG_DIR
from travel.avia.ad_feed.ad_feed.validation.k50 import K50Validator, ROWS_MAX_NUMBER
from travel.avia.library.python.lib_yt.client import configured_client

log = logging.getLogger(__name__)


@lru_cache(maxsize=None)
def create_validator() -> K50Validator:
    return K50Validator(rows_max_number=ROWS_MAX_NUMBER)


@lru_cache(maxsize=None)
def create_city_alias_supplier() -> CityAliasSupplier:
    return CityAliasSupplier(table_path=CityAliasSettings().table, client=create_yt_client())


class AppSettings(BaseSettings):
    redir_balance_log_age: int = 90
    cpa_orders_age: int = 90
    search_log_age: int = 90
    redir_balance_log_dir: str = REDIR_BALANCE_LOG_DIR
    search_log_dir: str = AVIA_USER_SEARCH_LOG_DIR
    stations_table: str = '//home/rasp/reference/station'
    stations_mapping_table = '//home/rasp/reference/station2settlement'
    airport_blacklist_table: str = '//home/avia/data/ad-feed/blacklist'


@click.group()
def main():
    logging.basicConfig(level=logging.INFO, stream=sys.stdout)


def set_by_env(
    ctx: click.core.Context, _: click.core.Parameter, environment: Optional[Environment]
) -> Optional[Environment]:
    if environment is None:
        return None
    ctx.default_map = {
        'cpa_orders_table': CPA_ORDER_TABLE_FOR_ENV[environment],
        'shared_flights_api_url': SHARED_FLIGHTS_API_URL_FOR_ENV[environment],
        'output_table': OUTPUT_TABLE_FOR_ENVIRONMENT[environment],
    }
    return environment


@main.command()
@click.option(
    '--env',
    'environment',
    type=EnumType(Environment, case_sensitive=False),
    required=False,
    is_eager=True,
    callback=set_by_env,
)
@click.option('--national-version', type=click.Choice(list(NationalVersion), case_sensitive=False), required=True)
@click.option('--output-table', type=str, required=True)
@click.option('--min-price-table', type=str, required=False)
@click.option('--cpa-orders-table', type=str, required=True)
@click.option('--shared-flights-api-url', type=str, required=True)
@click.option('--mds-s3-bucket-name', type=str, default='avia-indexer', required=True)
@click.option('--mds-s3-prefix', type=str, default='ad-feed', required=True)
@click.option('--mds-s3-key', type=str, default='ru.csv', required=True)
@click.option('--min-click-price', type=float, default=3, required=True)
def generate(
    national_version: NationalVersion,
    environment: Environment,
    output_table: str,
    min_price_table: str,
    shared_flights_api_url: str,
    cpa_orders_table: str,
    mds_s3_bucket_name: str,
    mds_s3_prefix: str,
    mds_s3_key: Optional[str],
    min_click_price: float,
):
    if environment is not None:
        min_price_table = MIN_PRICE_TABLE_FOR_ENVIRONMENT[environment].format(nv=national_version)
    if min_price_table is None:
        raise ValueError()

    app_settings = AppSettings()

    log.info(
        'Running dumper with min_price_table %s, redir_balance_log_dir %s, '
        'cpa_orders_table %s, shared_flights_api_url %s, search_log_dir %s, '
        'output_table %s',
        min_price_table,
        app_settings.redir_balance_log_dir,
        cpa_orders_table,
        shared_flights_api_url,
        app_settings.search_log_dir,
        output_table,
    )

    now = datetime.now()
    yt_client = create_yt_client()
    airport_blacklist = create_airport_blacklist(yt_client, app_settings)
    generator = create_direction_feed_generator(
        client=yt_client,
        min_price_table=min_price_table,
        redir_balance_log_dir=app_settings.redir_balance_log_dir,
        redir_balance_log_since=now - timedelta(app_settings.redir_balance_log_age),
        national_version=national_version,
        cpa_orders_table=cpa_orders_table,
        cpa_orders_since=now - timedelta(app_settings.cpa_orders_age),
        shared_flights_api_url=shared_flights_api_url,
        search_log_dir=app_settings.search_log_dir,
        search_log_since=now - timedelta(app_settings.search_log_age),
        airport_supplier=AirportInfoSupplier(
            yt_client=yt_client,
            stations_table=app_settings.stations_table,
            station2settlement_table=app_settings.stations_mapping_table,
        ),
        city_alias_supplier=create_city_alias_supplier(),
        intermediate_table=f'{output_table}-full',
        click_price_threshold=min_click_price,
        airport_blacklist=airport_blacklist,
    )
    YtDumper(client=yt_client).dump(generator.generate_feed(), output_table=output_table)
    s3_client = create_s3_client()
    s3_path = S3Path(key=mds_s3_key, prefix=mds_s3_prefix, bucket=mds_s3_bucket_name)
    dump_yt_to_s3_csv(
        yt_path=output_table,
        csv_path=s3_path,
        s3_client=s3_client,
        yt_client=yt_client,
        converter=create_direction_feed_converter(),
        validator=create_validator(),
    )
    send_file_metrics(path=s3_path, client=s3_client, reporter=get_solomon_reporter())


@lru_cache(maxsize=None)
def create_yt_client() -> YtClient:
    yt_settings = YtSettings()
    return configured_client(yt_settings.proxy, yt_settings.token)


@lru_cache(maxsize=None)
def create_s3_client() -> S3ClientProto:
    mds_settings = MdsSettings()
    return boto3.session.Session(
        aws_access_key_id=mds_settings.access_key_id,
        aws_secret_access_key=mds_settings.access_key_secret,
    ).client(
        service_name='s3',
        endpoint_url=mds_settings.endpoint,
    )


def create_airport_blacklist(yt_client: YtClient, app_settings: AppSettings) -> AirportBlacklist:
    return AirportBlacklist(
        yt_client=yt_client,
        blacklist_table=app_settings.airport_blacklist_table,
        stations_table=app_settings.stations_table,
    )


OUTPUT_TABLE_FOR_SNS = {
    Environment.TESTING: '//home/avia/testing/data/ad-feed/ru/city_and_station',
    Environment.PRODUCTION: '//home/avia/data/ad-feed/ru/city_and_station',
}


def set_by_env_for_sns(
    ctx: click.core.Context, _: click.core.Parameter, environment: Optional[Environment]
) -> Optional[Environment]:
    if environment is None:
        return None
    ctx.default_map = {
        'cpa_orders_table': CPA_ORDER_TABLE_FOR_ENV[environment],
        'shared_flights_api_url': SHARED_FLIGHTS_API_URL_FOR_ENV[environment],
        'output_table': OUTPUT_TABLE_FOR_SNS[environment],
    }
    return environment


@main.command()
@click.option(
    '--env',
    'environment',
    type=EnumType(Environment, case_sensitive=False),
    required=False,
    is_eager=True,
    callback=set_by_env_for_sns,
)
@click.option('--national-version', type=click.Choice(list(NationalVersion), case_sensitive=False), required=True)
@click.option('--output-table', type=str, required=True)
@click.option('--min-price-table', type=str, required=False)
@click.option('--cpa-orders-table', type=str, required=True)
@click.option('--shared-flights-api-url', type=str, required=True)
@click.option('--mds-s3-bucket-name', type=str, default='avia-indexer', required=True)
@click.option('--mds-s3-prefix', type=str, default='ad-feed', required=True)
@click.option('--mds-s3-key', type=str, default='settlements-and-stations.csv', required=False)
@click.option('--min-click-price', type=float, default=3, required=True)
def generate_sns(
    national_version: NationalVersion,
    environment: Environment,
    output_table: str,
    min_price_table: str,
    shared_flights_api_url: str,
    cpa_orders_table: str,
    mds_s3_bucket_name: str,
    mds_s3_prefix: str,
    mds_s3_key: Optional[str],
    min_click_price: float,
):
    if environment is not None:
        min_price_table = MIN_PRICE_TABLE_FOR_ENVIRONMENT[environment].format(nv=national_version)
    if min_price_table is None:
        raise ValueError()

    now = datetime.now()
    app_settings = AppSettings()
    yt_client = create_yt_client()
    airport_blacklist = create_airport_blacklist(yt_client, app_settings)
    generator = create_stations_and_settlements_generator(
        client=yt_client,
        min_price_table=min_price_table,
        redir_balance_log_dir=app_settings.redir_balance_log_dir,
        redir_balance_log_since=now - timedelta(app_settings.redir_balance_log_age),
        national_version=national_version,
        cpa_orders_table=cpa_orders_table,
        cpa_orders_since=now - timedelta(app_settings.cpa_orders_age),
        shared_flights_api_url=shared_flights_api_url,
        search_log_dir=app_settings.search_log_dir,
        search_log_since=now - timedelta(app_settings.search_log_age),
        stations_table=app_settings.stations_table,
        stations_mapping_table=app_settings.stations_mapping_table,
        intermediate_table=f'{output_table}-full',
        click_price_threshold=min_click_price,
        airport_blacklist=airport_blacklist,
    )
    YtDumper(client=yt_client).dump(generator.generate_feed(), output_table=output_table)
    s3_client = create_s3_client()
    s3_path = S3Path(key=mds_s3_key, prefix=mds_s3_prefix, bucket=mds_s3_bucket_name)
    dump_yt_to_s3_csv(
        yt_path=output_table,
        csv_path=s3_path,
        s3_client=s3_client,
        yt_client=yt_client,
        converter=create_stations_and_settlements_feed_converter(),
        validator=create_validator(),
    )
    send_file_metrics(path=s3_path, client=s3_client, reporter=get_solomon_reporter())


if __name__ == '__main__':
    main()
