import datetime
import logging
import os
import tempfile
from typing import Optional

import boto3
import click
import sys

from pydantic import BaseSettings
from yt.wrapper import YtClient

from travel.avia.ad_feed.ad_feed.airport_blacklist import AirportBlacklist
from travel.avia.ad_feed.ad_feed.feed_generator.factory import get_solomon_reporter
from travel.avia.ad_feed.ad_feed.metrics import send_file_metrics
from travel.avia.ad_feed.ad_feed.feed_generator import create_wizard_like_generator
from travel.avia.ad_feed.ad_feed.runner.tools import EnumType
from travel.avia.ad_feed.ad_feed.converter.factory import create_wizard_like_converter
from travel.avia.ad_feed.ad_feed.converter.runner import upload_to_s3, S3_ENDPOINT_FOR_ENV, S3Path
from travel.avia.ad_feed.ad_feed.environment import Environment
from travel.avia.ad_feed.ad_feed.feed_generator.destination_only import TRAVEL_HOST_BY_ENV
from travel.avia.ad_feed.ad_feed.feed_generator.wizard_like import (
    WizardClient,
    WizardSettings,
    OrderUrlBuilder,
)
from travel.avia.library.python.lib_yt.client import configured_client

logger = logging.getLogger(__name__)


class AppSettings(BaseSettings):
    stations_table: str = '//home/rasp/reference/station'
    airport_blacklist_table: str = '//home/avia/data/ad-feed/blacklist'


@click.group()
def main():
    logging.basicConfig(level=logging.INFO, stream=sys.stdout)


WIZARD_URL_BY_ENV = {
    Environment.TESTING: 'http://wizard.testing.avia.yandex.net/',
    Environment.PRODUCTION: 'http://wizard.production.avia.yandex.net/',
}


def set_by_env(
    ctx: click.core.Context, _: click.core.Parameter, environment: Optional[Environment]
) -> Optional[Environment]:
    if environment is None:
        return None
    ctx.default_map = {
        'wizard_url': WIZARD_URL_BY_ENV[environment],
        'mds_s3_endpoint': S3_ENDPOINT_FOR_ENV[environment],
    }
    return environment


def create_airport_blacklist(yt_client: YtClient, app_settings: AppSettings) -> AirportBlacklist:
    return AirportBlacklist(
        yt_client=yt_client,
        blacklist_table=app_settings.airport_blacklist_table,
        stations_table=app_settings.stations_table,
    )


@main.command()
@click.option(
    '--env',
    'environment',
    type=EnumType(Environment, case_sensitive=False),
    required=True,
    is_eager=True,
    callback=set_by_env,
)
@click.option('--yt-proxy', type=str, required=True)
@click.option('--yt-token', default=lambda: os.getenv('YT_TOKEN'), type=str, required=True)
@click.option('--wizard-url', type=str, required=True)
@click.option('--mds-access-key-id', type=str, default=lambda: os.getenv('MDS_ACCESS_KEY_ID'), required=True)
@click.option('--mds-access-key-secret', type=str, default=lambda: os.getenv('MDS_ACCESS_KEY_SECRET'), required=True)
@click.option('--mds-s3-endpoint', type=str, default='https://s3.mdst.yandex.net', required=True)
@click.option('--mds-s3-bucket-name', type=str, default='avia-indexer', required=True)
@click.option('--mds-s3-prefix', type=str, default='ad-feed', required=True)
@click.option('--mds-s3-key', type=str, default='yandex-smart-banners.xml', required=True)
@click.option('--request-delay', type=int, default=50, required=True, help='Time between requests in ms')
def generate(
    yt_proxy: str,
    yt_token: str,
    environment: Environment,
    wizard_url: str,
    mds_access_key_id: str,
    mds_access_key_secret: str,
    mds_s3_endpoint: str,
    mds_s3_bucket_name: str,
    mds_s3_prefix: str,
    mds_s3_key: str,
    request_delay: int,
) -> None:
    app_settings = AppSettings()
    yt_client = configured_client(yt_proxy, yt_token)
    wizard_client = WizardClient(settings=WizardSettings(url=wizard_url, variants_path='api/cached_variants/'))
    airport_blacklist = create_airport_blacklist(yt_client, app_settings)
    generator = create_wizard_like_generator(
        request_delay=datetime.timedelta(milliseconds=request_delay),
        url_builder=OrderUrlBuilder(host=TRAVEL_HOST_BY_ENV[environment]),
        wizard_client=wizard_client,
        yt_client=yt_client,
        airport_blacklist=airport_blacklist,
    )
    converter = create_wizard_like_converter()
    s3_client = boto3.session.Session(
        aws_access_key_id=mds_access_key_id,
        aws_secret_access_key=mds_access_key_secret,
    ).client(
        service_name='s3',
        endpoint_url=mds_s3_endpoint,
    )
    remote_path = S3Path(prefix=mds_s3_prefix, key=mds_s3_key, bucket=mds_s3_bucket_name)
    with tempfile.NamedTemporaryFile(mode='wb') as output_file:
        logger.info('Dumping results to local temp file %s', output_file.name)
        converter.convert((v for v in generator.generate_feed()), output_file)
        logger.info('Done. File size is %d', os.path.getsize(output_file.name))

        upload_to_s3(remote_path, output_file.name, s3_client)
    send_file_metrics(path=remote_path, client=s3_client, reporter=get_solomon_reporter())


if __name__ == '__main__':
    main()
