from typing import IO, Any, Final
import csv


class ValidationError(Exception):
    pass


ROWS_MAX_NUMBER: Final = 10 ** 5


class K50Validator:
    def __init__(self, rows_max_number: int):
        self.rows_max_number = rows_max_number

    def validate(self, data: IO[Any]) -> None:
        reader = csv.DictReader(
            data,
            restval='',
            dialect='excel',
            delimiter=',',
            quoting=csv.QUOTE_ALL,
            quotechar='"',
        )
        if reader.fieldnames is None:
            raise ValidationError('No field names')
        if 'url' not in reader.fieldnames:
            raise ValidationError('url is not presented in field names')
        key_fields = []
        for field in reader.fieldnames:
            if field.endswith('*'):
                key_fields.append(field)
        if not key_fields:
            key_fields = ['id']
            if 'id' not in reader.fieldnames:
                raise ValidationError('No id column')
        key_set = set()
        rows_number = 0
        for row in reader:
            key_list = []
            for field in key_fields:
                key_list.append(row[field])
            key = tuple(key_list)
            if key in key_set:
                raise ValidationError(f'Key duplicate: {key}')
            key_set.add(key)
            rows_number += 1
        if rows_number > self.rows_max_number:
            raise ValidationError(f'Row limit exceeded: file contains {rows_number} rows')
