import logging
import subprocess
from pathlib import Path
from typing import Optional, Final, Any
import os

import click
import shlex
import sys
import yaml
import enum

from library.python.vault_client.instances import Production as VaultClient
from pydantic import BaseModel, Field
from sandbox.common.auth import OAuth
from sandbox.common.rest import Client as SandboxClient

from library.python import resource
from travel.avia.ad_feed.ad_feed.runner.tools import EnumType
from travel.avia.ad_feed.ad_feed.environment import Environment


@click.group()
def main():
    logging.basicConfig(level=logging.INFO, stream=sys.stdout)


class Schedule(BaseModel):
    fail_on_error: bool
    retry: dict
    sequential_run: bool
    repetition: dict


class PriorityClass(str, enum.Enum):
    BACKGROUND = 'BACKGROUND'
    USER = 'USER'
    SERVICE = 'SERVICE'


class PrioritySubclass(str, enum.Enum):
    LOW = 'LOW'
    NORMAL = 'NORMAL'
    HIGH = 'HIGH'


class Priority(BaseModel):
    class_: PriorityClass = Field(PriorityClass.BACKGROUND, alias='class')
    subclass: PrioritySubclass = PrioritySubclass.NORMAL

    class Config:
        allow_population_by_field_name = True


class Config(BaseModel):
    cmd: str
    env: dict[str, str]
    path: str
    schedule: Schedule
    description: Optional[str] = None
    priority: Priority = Priority()


def fetch_secrets(config: Config) -> None:
    vault_client = VaultClient(decode_files=True)
    for key, env_value in config.env.items():
        if not env_value.startswith('sec-'):
            continue
        secret_id, field = env_value.split(':')
        config.env[key] = vault_client.get_version(secret_id)['value'][field]


def get_config(name: str, env: str) -> Config:
    return Config.parse_obj(yaml.safe_load(resource.find('/tasks.yaml'))[name][env])


def _get_avia_task_by_name(name: str) -> str:
    return f'ad_feed:{name}'


_RESOURCE_TYPE: Final = 'OTHER_RESOURCE'


def _build_and_upload(name: str, env: Environment, path_to_binary: Path) -> None:
    subprocess.run(
        f'cd {shlex.quote(str(path_to_binary.parent))} && '
        f'ya make -r --yt-store && '
        f'ya upload {shlex.quote(path_to_binary.name)} -A release_type={shlex.quote(_RELEASE_TYPE_BY_ENV[env])} '
        f'-A avia_task={shlex.quote(_get_avia_task_by_name(name))} --ttl=inf --sandbox --type={shlex.quote(_RESOURCE_TYPE)}',
        shell=True,
        stdout=sys.stdout,
        stderr=sys.stderr,
    )


@main.command(help='Compile and upload binary file to sandbox as a resource')
@click.option('--name', type=str, required=True)
@click.option('--env', type=EnumType(Environment, case_sensitive=False), required=True)
@click.option('--root', type=click.Path(), help='Path to project root')
def upload(name: str, env: Environment, root: str):
    config = get_config(name, env.value)
    path_to_binary = Path(root) / config.path
    _build_and_upload(name=name, env=env, path_to_binary=path_to_binary)


_RELEASE_TYPE_BY_ENV = {
    Environment.TESTING: 'testing',
    Environment.PRODUCTION: 'stable',
    Environment.PREPRODUCTION: 'prestable',
}

ALERT_ACCOUNTS: Final = ['avia-alerts']
OWNER: Final = 'AVIA'
AUTHOR: Final = 'robot-avia-api-pilot'
TASK_TYPE: Final = 'RUN_BINARY_RESOURCE'


def _set_run_params(fields: list[dict[str, Any]], config: Config, name: str, env: Environment) -> None:
    value_by_title = {
        'env_vars': config.env,
        'cmd': config.cmd,
        'resource_attrs': {'avia_task': _get_avia_task_by_name(name), 'release_type': _RELEASE_TYPE_BY_ENV[env]},
        'resource_type': _RESOURCE_TYPE,
        'resource_by_id': False,
        'unpack resource': False,
    }
    for field in fields:
        title = field['title']
        if title in value_by_title:
            field['value'] = value_by_title[title]


def _get_tags(name: str, env: Environment) -> list[str]:
    return ['AD_FEED', env.value.upper(), f'AVIA_TASK_AD_FEED_{name.upper()}']


def _get_or_create_scheduler(name: str, env: Environment, sandbox_client: SandboxClient) -> int:
    schedulers = sandbox_client.scheduler[
        {'task_type': TASK_TYPE, 'tags': _get_tags(name, env), 'all_tags': True, 'limit': 100}
    ]['items']
    if len(schedulers) > 1:
        click.echo("Error: More then one suitable scheduler")
        raise exit(1)
    if len(schedulers) == 0:
        schedulers = [sandbox_client.scheduler.create(task_type=TASK_TYPE)]
        click.echo(f'New scheduler with id: {schedulers[0]["id"]} is created')
    return schedulers[0]['id']


def _update_scheduler(
    scheduler_id: int, name: str, env: Environment, config: Config, sandbox_client: SandboxClient
) -> None:
    task_data = sandbox_client.scheduler[scheduler_id].read()['task']
    task_data['tags'] = _get_tags(name, env)
    task_data['owner'] = OWNER
    task_data['notifications'] = [
        {'check_status': None, 'transport': 'email', 'recipients': ALERT_ACCOUNTS, 'statuses': ['BREAK', 'FAILURE']}
    ]
    task_data['description'] = config.description or ''
    task_data['priority'] = config.priority.dict(by_alias=True)
    _set_run_params(task_data['custom_fields'], config=config, name=name, env=env)
    sandbox_client.scheduler[scheduler_id].update(
        task=task_data,
        schedule=config.schedule.dict(),
        owner=OWNER,
        author=AUTHOR,
        scheduler_notifications=[
            {'check_status': None, 'transport': 'email', 'recipients': ALERT_ACCOUNTS, 'statuses': ['FAILURE']}
        ],
    )


def get_sandbox_token() -> str:
    token_file_path = Path.home() / ".sandbox_token"
    try:
        with open(token_file_path, 'r') as f:
            return f.read()
    except FileNotFoundError:
        token = click.prompt('Перейдите на страницу https://sandbox.yandex-team.ru/oauth и скопируйте токен', type=str)
        with open(token_file_path, 'w') as f:
            f.write(token)
        os.chmod(token_file_path, 0o600)
        click.echo(f'Токен сохранён в {token_file_path}')
        return token


def _get_sandbox_client() -> SandboxClient:
    token = get_sandbox_token()
    return SandboxClient(auth=OAuth(token))


@main.command(help='Deploy all tasks')
@click.option('--env', type=EnumType(Environment, case_sensitive=False), required=True)
@click.option('--root', type=click.Path(), help='Path to project root', required=True)
def deploy_all(env: Environment, root: str) -> None:
    config = yaml.safe_load(resource.find('/tasks.yaml'))
    sandbox_client = _get_sandbox_client()
    for name in config:
        if env.value not in config[name]:
            continue
        current = Config.parse_obj(config[name][env.value])

        _deploy_generator(name=name, root=root, config=current, env=env, sandbox_client=sandbox_client)


@main.command(help='Deploy scheduler')
@click.option('--name', type=str, required=True)
@click.option('--env', type=EnumType(Environment, case_sensitive=False), required=True)
@click.option('--root', type=click.Path(), help='Path to project root', required=True)
def deploy(name: str, env: Environment, root: str):
    config = get_config(name, env.value)

    token = get_sandbox_token()
    sandbox_client = SandboxClient(auth=OAuth(token))

    _deploy_generator(name=name, root=root, config=config, env=env, sandbox_client=sandbox_client)


def _deploy_generator(name: str, root: str, config: Config, env: Environment, sandbox_client: SandboxClient) -> None:
    click.echo(f'Building {name}')
    path_to_binary = Path(root) / config.path
    _build_and_upload(name=name, env=env, path_to_binary=path_to_binary)

    click.echo(f'Deploying {name}')
    _deploy_scheduler(name=name, env=env, config=config, sandbox_client=sandbox_client)
    click.echo(f'{name} successfully deployed')


def _deploy_scheduler(name: str, env: Environment, config: Config, sandbox_client: SandboxClient) -> None:
    scheduler_id = _get_or_create_scheduler(name=name, env=env, sandbox_client=sandbox_client)
    _update_scheduler(scheduler_id=scheduler_id, name=name, env=env, config=config, sandbox_client=sandbox_client)
    sandbox_client.batch.schedulers.start.update(id=scheduler_id)
    click.echo(f'Scheduler https://sandbox.yandex-team.ru/scheduler/{scheduler_id}/view is launched')


@main.command(help='Run the task locally')
@click.option('--name', type=str, required=True)
@click.option('--env', type=EnumType(Environment, case_sensitive=False), required=True)
@click.option('--root', type=click.Path(), required=True, help='Path to project root')
def run(name: str, env: Environment, root: str):
    data = get_config(name, env.value)
    fetch_secrets(data)
    env_config = ' '.join(f'{shlex.quote(name)}={shlex.quote(value)}' for name, value in data.env.items())
    cmd = env_config + ' ' + data.cmd.format(resource=shlex.quote(str(Path(root) / data.path)))
    click.echo(cmd)
    subprocess.run(cmd, shell=True, stdout=sys.stdout, stderr=sys.stderr)


if __name__ == '__main__':
    main()
