import logging

from django import template
from django.conf import settings
from django.contrib.admin.helpers import AdminField, AdminReadonlyField

from travel.avia.admin.avia.sync_rasp.import_rasp_info_schema import (
    IMPORT_MODELS, field_name, model_path, NEW_RASP_SYNC_MODELS,
)

log = logging.getLogger(__name__)

register = template.Library()


FIELDS_BY_MODEL = {
    j['model']: {
        field_name(f): f for f in j['fields']
    }
    for j in IMPORT_MODELS
}


def field_path(model, object_id, fieldname):
    return '{}[{}].{}'.format(
        model_path(model),
        object_id,
        fieldname
    )


def field_raspadmin_url(model, object_id, fieldname):
    return 'https://{}/admin/{}/{}/{}/#{}'.format(
        settings.RASPADMIN_HOST,
        model._meta.app_label,
        model.__name__.lower(),
        object_id,
        fieldname
    )


@register.inclusion_tag('admin/avia/is_sync_field_comment.html', takes_context=True)
def is_sync_field_comment(context, adminfield):
    try:
        if isinstance(adminfield, AdminField):
            boundfield = adminfield.field
            model = boundfield.form.Meta.model
            fieldname = boundfield.name
        elif isinstance(adminfield, AdminReadonlyField):
            boundfield = adminfield.field
            model = adminfield.form.Meta.model
            fieldname = boundfield['name']
        else:
            return {}

        sync_field = FIELDS_BY_MODEL.get(model, {}).get(fieldname)

        is_syncable = sync_field is not None

        ctx = {
            'model': model,
            'fieldname': fieldname,
            'is_syncable': is_syncable and model in NEW_RASP_SYNC_MODELS,
        }

        object_id = context.get('object_id')

        if is_syncable:
            ctx.update({
                'sync_url': field_raspadmin_url(model, object_id, fieldname),
                'sync_title': field_path(model, object_id, fieldname),
            })

        return ctx

    except Exception:
        log.exception('is_sync_field_comment')

    return {}
