# -*- coding: utf-8 -*-

import logging
import json
import re
import shutil
import time
import os
import os.path
import zipfile

from email.header import Header
from codecs import open
from datetime import datetime, date, timedelta
from traceback import format_exc
from itertools import islice
import pytils.dt as russian_dt

from chardet.universaldetector import UniversalDetector

from django.apps import apps
from django.conf import settings
from django.contrib import messages
from django.db.models import Count, Sum
from django.template import RequestContext
from django.contrib.auth.decorators import login_required, permission_required
from django.contrib.admin import site
from django.contrib.admin.widgets import AdminDateWidget, ForeignKeyRawIdWidget
from django.shortcuts import get_object_or_404, render_to_response as django_render_to_response
from django.http import HttpResponse, HttpResponseRedirect, Http404
from django.utils.html import escape
from django.utils.text import Truncator
from django.utils.safestring import mark_safe, mark_for_escaping
from django import forms
from django.core.exceptions import PermissionDenied
from django.core.urlresolvers import reverse

from travel.avia.library.python.common.utils import dateparser
from travel.avia.library.python.common.utils.date import fix_date
from travel.avia.library.python.common.utils.jsonutils import serialize
from travel.avia.library.python.common.utils.httpresponses import jsonp_response
from travel.avia.library.python.common.utils.unicode_csv import UnicodeDictReader
from travel.avia.library.python.common.models.geo import District, Country, Settlement, Region, Station, Point, StationPhone, ExternalDirection, Direction  # noqa
from travel.avia.library.python.common.models.schedule import RThread, Supplier
from travel.avia.library.python.common.models.transport import TrainPseudoStationMap
from travel.avia.admin.lib.exceptions import SimpleUnicodeException
from travel.avia.admin.lib.jinja import render_to_string
from travel.avia.admin.lib.tmpfiles import clean_temp, get_tmp_filepath
from travel.avia.admin.lib.unzip import unpack_zip_file
from travel.avia.admin.order.models import Partner, StatisticsEntry
from travel.avia.admin.www.utils.common import first
from travel.avia.admin.www.utils.data import items_tds


log = logging.getLogger(__name__)


# quick tree zone
@login_required
def quick_tree_planet(request, id=None):
    """ Страница планеты быстрого спуска по гео-дереву """
    context = {
        'sub_classes': [{
            'objects_rows': items_tds(Country.objects.all(), 4),
            'object_type': 'country',
            'not_lowest': 1,
            'add_object_params': '',
            'title': u'Страны'
        }]
    }
    return django_render_to_response('admin/quick_tree.html', context, RequestContext(request))


@login_required
def quick_tree_country(request, id):
    """ Страница страны быстрого спуска по гео-дереву """
    try:
        country = Country.objects.get(id=id)
    except Country.DoesNotExist:
        id = 225
        country = Country.objects.get(id=id)

    regions = Region.objects.filter(country=country)
    settlements = Settlement.objects.filter(country=country, region__isnull=True)
    stations = Station.objects.filter(hidden=False, country=country, region__isnull=True, settlement__isnull=True)

    context = {'current_node': country,
               'sub_classes': [{'objects_rows': items_tds(regions),
                                'object_type': 'region',
                                'not_lowest': 1,
                                'add_object_params': 'country_id=' + str(id),
                                'title': u'Области'},
                               {'objects_rows': items_tds(settlements),
                                'object_type': 'settlement',
                                'not_lowest': 1,
                                'add_object_params': 'country_id=' + str(id),
                                'title': u'Города'},
                               {'objects_rows': items_tds(stations),
                                'object_type': 'station',
                                'not_lowest': 1,
                                'add_object_params': 'country_id=' + str(id),
                                'title': u'Станции'}
                               ]
               }

    return django_render_to_response('admin/quick_tree.html', context, RequestContext(request))


@login_required
def quick_tree_region(request, id):
    """ Страница области быстрого спуска по гео-дереву """
    context = {}
    try:
        region = Region.objects.get(id=id)
        settlements = Settlement.objects.filter(region=region).\
                                 order_by('majority__id', 'title')
        settlements = map(
            lambda o: {
                'id': o.id,
                'title': o.title,
                'style': (
                    o.majority
                    and (
                        (o.majority.id in [1, 2] and 'font-weight: bold;')
                        or (o.majority.id in [5] and 'font-size: 9px; font-style: Italic;')
                        or ''
                    )
                    or ''
                )
            },
            settlements
        )
    except Region.DoesNotExist:
        region = None
        settlements = []
    country_id = None
    if region and region.country:
        country_id = region.country.id

    sub_classes = []
    sub_classes.append({
        'objects_rows': items_tds(settlements),
        'object_type': 'settlement',
        'not_lowest': 1,
        'add_object_params': 'region_id=' + str(id) + (country_id and ("&country_id=" + str(country_id)) or ""),
        'title': u'Города'
    })

    stations = []
    if region:
        stations = Station.objects.filter(hidden=False, region=region, settlement__isnull=True).order_by('majority__id', 'title')
        stations = map(
            lambda o: {
                'id': o.id,
                'title': o.title,
                'code': o.t_type.code,
                'style': (
                    o.majority
                    and (
                        (o.majority.id in [1] and 'font-weight: bold;')
                        or (o.majority.id in [3] and 'font-size: 9px; font-style: Italic;')
                        or ''
                    ) or ''
                )
            },
            stations
        )
        sub_classes.append({
            'objects_rows': items_tds(stations),
            'object_type': 'station',
            'not_lowest': 1,
            'add_object_params': 'region_id=' + str(region.id),
            'title': u'Станции'
        })
    context['current_node'] = region
    context['sub_classes'] = sub_classes

    return django_render_to_response('admin/quick_tree.html', context, RequestContext(request))


@login_required
def quick_tree_settlement(request, id):
    """ Страница города быстрого спуска по гео-дереву """
    context = {}
    try:
        settlement = Settlement.objects.get(id=id)
        stations = Station.objects.filter(settlement=settlement).order_by('majority__id', 'title')
        stations = map(
            lambda o: {
                'id': o.id,
                'title': o.title,
                'icon': '%s_small.gif' % o.t_type.code,
                'style': (
                    o.majority
                    and (
                        (o.majority.id in [1] and 'font-weight: bold;')
                        or (o.majority.id in [3] and 'font-size: 9px; font-style: Italic;')
                        or ''
                    )
                    or ''
                )
            },
            stations
        )
    except Settlement.DoesNotExist:
        settlement = None
        stations = []

    sub_classes = [{
        'objects_rows': items_tds(stations),
        'object_type': 'station',
        'not_lowest': 1,
        'add_object_params': 'settlement_id=' + str(id),
        'title': u'Станции'
    }]
    context['current_node'] = settlement
    context['sub_classes'] = sub_classes

    return django_render_to_response('admin/quick_tree.html', context, RequestContext(request))


@login_required
def quick_tree_station(request, id):
    """ Страница области быстрого спуска по гео-дереву """
    context = {}
    try:
        station = Station.objects.get(hidden=False, id=id)
        objects = StationPhone.objects.filter(station=station)
        objects = map(lambda o: {'id': o.id, 'title': o.phone + (o.note and u' (%s)' % o.note)}, objects)
    except Station.DoesNotExist:
        station = None
        objects = []
    sub_classes = [{
        'objects_rows': items_tds(objects),
        'object_type': 'stationphone',
        'not_lowest': 0,
        'add_object_params': 'station_id=' + str(id),
        'title': u'Телефоны станции'
    }]
    context['current_node'] = station
    context['sub_classes'] = sub_classes

    return django_render_to_response('admin/quick_tree.html', context, RequestContext(request))


@login_required
def show_file(request, app_label, model, field_name, id):
    model = apps.get_model(app_label, model)
    if model and request.user.has_perm(u"%s.change_%s" % (app_label,
                                                          model.__name__.lower())):
        instance = model.objects.get(id=id)
        content = getattr(instance, field_name)
        encoding = content.encoding
        filename = getattr(instance, '%s_name' % field_name, None)
        response = HttpResponse(content.encode(encoding), content_type="text/plain; charset=%s" % encoding)

        if filename:
            response['Content-Disposition'] = 'attachment; filename=%s' % str(Header(filename, encoding))

        return response
    else:
        HttpResponseRedirect("/admin/")


@login_required
def show_log(request, name):
    log_path = os.path.join(settings.LOG_PATH, 'special/admin_run', name)
    if not os.path.exists(log_path):
        raise Http404("Log file %s was not found")

    f = open(log_path, "r", "utf8", errors="ignore")
    content = f.read()
    f.close()
    return HttpResponse(content, content_type="text/plain; charset=utf8")


@login_required
def show_run_log_list(request):
    log_dir_path = os.path.join(settings.LOG_PATH, 'special/admin_run')

    log_names = []

    for log_name in os.listdir(log_dir_path):
        if log_name.endswith(".log") and os.path.isfile(os.path.join(log_dir_path, log_name)):
            log_names.append(log_name)

    def filter_by_current_instance(log_names):
        return filter(lambda x: x.split('_')[2] == settings.INSTANCE_ROLE.code, log_names)

    log_names = filter_by_current_instance(log_names)

    logs = []

    for log_name in log_names:
        dt = u" ".join(log_name.split('_')[:2])
        try:
            dt = datetime(*time.strptime(dt, u"%Y-%m-%d %H.%M.%S")[:6])
        except ValueError:
            continue

        log_file = open(os.path.join(log_dir_path, log_name), 'r', encoding='utf-8')
        first_lines = list(islice(log_file, 10))
        log_file.close()

        logs.append((dt, log_name, u"".join(first_lines)))

    logs.sort(reverse=True)

    logs = logs[:30]

    return django_render_to_response('admin/log_list.html',
                                     {'logs': logs, 'last_log_name': logs and logs[0][1]},
                                     RequestContext(request))


@login_required
def show_run_log_list_export(request):
    log_dir_path = os.path.join(settings.LOG_PATH, 'special/admin_run')

    log_names = []

    reg = re.compile(ur'cache_long_haul')

    for log_name in os.listdir(log_dir_path):
        if log_name.endswith(".log")\
           and os.path.isfile(os.path.join(log_dir_path, log_name))\
           and bool(reg.search(log_name)):
            log_names.append(log_name)

    def filter_by_current_instance(log_names):
        return filter(lambda x: x.split('_')[2] == settings.INSTANCE_ROLE.code, log_names)

    log_names = filter_by_current_instance(log_names)

    logs = []

    for log_name in log_names:
        dt = u" ".join(log_name.split('_')[:2])
        try:
            dt = datetime(*time.strptime(dt, u"%Y-%m-%d %H.%M.%S")[:6])
        except ValueError:
            continue

        log_file = open(os.path.join(log_dir_path, log_name), 'r', encoding='utf-8')
        first_lines = log_file.readlines()[:10]
        log_file.close()

        logs.append((dt, log_name, u"".join(first_lines)))

    logs.sort(reverse=True)

    logs = logs[:30]

    return django_render_to_response('admin/log_list.html',
                                     {'logs': logs, 'last_log_name': logs and logs[0][1]},
                                     RequestContext(request))


@login_required
@permission_required("www.add_rthread")
def copy_thread(request):
    thread_id = request.POST.get('thread_id')
    try:
        thread = RThread.objects.get(id=thread_id)
    except RThread.DoesNotExist:
        return HttpResponseRedirect('/admin/www/rthread/%s' % thread_id)

    thread = thread.copy()
    messages.add_message(request, messages.SUCCESS, u"Нитка скопирована")
    return HttpResponseRedirect('/admin/www/rthread/%d' % thread.id)


class UploadZipForm(forms.Form):
    file_encoding = 'cp1251'

    zip_file = forms.FileField(required=True)

    def clean_zip_file(self):
        if 'zip_file' not in self.cleaned_data:
            return self.cleaned_data

        try:
            zip_file = zipfile.ZipFile(self.cleaned_data['zip_file'])

            names = zip_file.namelist()

            detector = UniversalDetector()

            not_unicode_file_count = 0
            for name in names:
                if not isinstance(name, unicode):
                    detector.feed(name)
                    not_unicode_file_count += 1

            detector.close()

            if not_unicode_file_count and detector.result['confidence'] < 0.5:
                raise forms.ValidationError(u"Непонятная кодировка имен файлов")

            name_encoding = detector.result['encoding']

            files = {}

            for name in names:
                unicode_name = name if isinstance(name, unicode) else name.decode(name_encoding)

                if self.file_encoding != 'bin':
                    try:
                        files[unicode_name] = zip_file.read(name).decode(self.file_encoding)
                    except UnicodeDecodeError:
                        raise forms.ValidationError(u"Не смогли декодировать файл %s из %s" % (unicode_name, self.file_encoding))

            self.cleaned_data['files'] = files

            if self.file_encoding != 'bin' and not files:
                raise forms.ValidationError(u"Архив не содержит файлов")

            return self.cleaned_data['zip_file']

        except zipfile.error:
            raise forms.ValidationError(u"Неправильный zip файл")


@login_required
@permission_required("www.change_externaldirection")
def load_schemas_zip_file(request):
    if request.method == "POST":
        form = UploadZipForm(request.POST, request.FILES)

        if form.is_valid():
            missing = []

            code_re = re.compile(ur'([\w-]+)\.', re.U)

            for name, data in form.cleaned_data['files'].items():
                m = code_re.search(name)

                if not m:
                    continue

                code = m.group(1)

                try:
                    direction = ExternalDirection.objects.get(code=code)
                    direction.schema_file = data
                    direction.save()
                except ExternalDirection.DoesNotExist:
                    missing.append(name)

            message = u"Файл со схемами направлений загружен"

            if missing:
                message += u'. Не найдены направления для файлов: %s' % ", ".join(missing)

            messages.add_message(request, messages.SUCCESS, mark_safe(message))

            return HttpResponseRedirect("/admin/www/externaldirection/load_zip_file")
    else:
        form = UploadZipForm()
    context = {'form': form,
               'title': u"Загрузить zip файл со схемами направлений",
               'opts': ExternalDirection._meta}
    return django_render_to_response('admin/www/load_zip_file_form.html', context, RequestContext(request))


class BinUploadZipForm(UploadZipForm):
    file_encoding = 'bin'


@login_required
def recalc_static_pages(request):
    from travel.avia.admin.staticpages.models import StaticPage
    StaticPage.recalc_materialized_paths()

    return HttpResponseRedirect(request.META['HTTP_REFERER'])


@login_required
@permission_required("www.change_externaldirection")
def externaldirection_edit_schema(request, pk):
    direction = get_object_or_404(ExternalDirection.objects, pk=pk)

    if request.method == 'POST':
        direction.parse_schema_json(request.POST['schema'])

        direction.save()

        if '_save' in request.POST:
            return HttpResponseRedirect('../')

    context = {
        'title': u'Просмотр / редактирование схемы направления',
        'original': direction,
        'opts': direction._meta,
        'schema': render_to_string('scheme/scheme.html', {
            'edit': True,
            'schema': direction.schema(edit=True),
            }, request=request),
    }

    return django_render_to_response('admin/www/externaldirection/edit_schema.html', context, RequestContext(request))


@permission_required("www.change_externaldirection")
@jsonp_response
def externaldirection_stations(request, pk):
    direction = ExternalDirection.objects.get(pk=pk)

    rv = [(s.title, s.get_schedule_url(type_='suburban', direction=direction.code)) for s in direction.stations]

    return rv


class RawIdWidget(forms.TextInput):
    """
    A Widget for displaying ForeignKeys in the "raw_id" interface rather than
    in a <select> box.
    """
    def __init__(self, model, attrs=None):
        self.model = model
        super(RawIdWidget, self).__init__(attrs)

    def render(self, name, value, attrs=None):
        if attrs is None:
            attrs = {}
        related_url = '/admin/%s/%s/' % (self.model._meta.app_label,
                                           self.model._meta.object_name.lower())
        params = self.url_parameters()
        if params:
            url = '?' + '&amp;'.join(['%s=%s' % (k, v) for k, v in params.items()])
        else:
            url = ''
        if 'class' not in attrs:
            attrs['class'] = 'vForeignKeyRawIdAdminField'  # The JavaScript looks for this hook.
        output = [super(RawIdWidget, self).render(name, value, attrs)]
        # TODO: "id_" is hard-coded here. This should instead use the correct
        # API to determine the ID dynamically.
        output.append('<a href="%s%s" class="related-lookup" id="lookup_id_%s" onclick="return showRelatedObjectLookupPopup(this);"> ' %
                      (related_url, url, name))
        output.append('<img src="%simg/admin/selector-search.gif" width="16" height="16" alt="%s" /></a>' % (settings.ADMIN_MEDIA_PREFIX, 'Lookup'))
        if value:
            output.append(self.label_for_value(value))
        return mark_safe(u''.join(output))

    def base_url_parameters(self):
        params = {}
        return params

    def url_parameters(self):
        from django.contrib.admin.views.main import TO_FIELD_VAR
        params = self.base_url_parameters()
        params.update({TO_FIELD_VAR: "pk"})
        return params

    def label_for_value(self, value):
        key = "pk"
        obj = self.model._default_manager.get(**{key: value})
        return '&nbsp;<strong>%s</strong>' % escape(Truncator(obj).words(14))


class ScheduleLeafsetForm(forms.Form):
    external_direction = forms.ModelChoiceField(queryset=ExternalDirection.objects.all(),
                                                required=False)
    station1 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station2 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station3 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station4 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station5 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station6 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    date1 = forms.DateField(required=False, widget=AdminDateWidget)
    date2 = forms.DateField(required=False, widget=AdminDateWidget)
    date3 = forms.DateField(required=False, widget=AdminDateWidget)
    date4 = forms.DateField(required=False, widget=AdminDateWidget)
    date5 = forms.DateField(required=False, widget=AdminDateWidget)
    date6 = forms.DateField(required=False, widget=AdminDateWidget)

    def clean(self):
        stations = []
        dates = []
        for n in range(1, 7):
            if self.cleaned_data.get('station%s' % n):
                stations.append(self.cleaned_data.get('station%s' % n))
        for n in range(1, 7):
            if self.cleaned_data.get('date%s' % n):
                dates.append(self.cleaned_data.get('date%s' % n))
        if not dates:
            raise forms.ValidationError(u"Нужно выбрать хотя бы одну дату")
        if not self.cleaned_data['external_direction'] and not stations:
            raise forms.ValidationError(u"Нужно выбрать хотя бы одну станцию или внешнее направление")
        elif self.cleaned_data['external_direction'] and stations:
            raise forms.ValidationError(u"Нужно выбрирать либо станцию, либо направления")

        self.cleaned_data['dates'] = dates
        self.cleaned_data['stations'] = stations
        return self.cleaned_data


@login_required
def upload_schedule_file(request):
    # Поставщики. Код для формы, название, имя файла
    suppliers = filter(lambda s: s.filename, Supplier.objects.filter(filename__isnull=False))
    full_path = os.path.normpath(os.path.join(settings.DATA_PATH, 'upload'))

    context = {
        'full_path': full_path,
        'suppliers': suppliers,
        'files': get_files_info(full_path)
    }

    if os.path.exists(full_path):
        if request.FILES:
            supplier = first(lambda s: s.code==request.POST['supplier'], suppliers)
            f = request.FILES['file']
            destination = open(os.path.join(full_path, supplier.filename), 'wb+')
            for chunk in f.chunks():
                destination.write(chunk)
            destination.close()
            messages.add_message(
                request, messages.SUCCESS,
                mark_safe(u"Загрузили файл расписаний {} поставщика {} {}".format(
                    supplier.filename, supplier.title, supplier.code))
            )
            return HttpResponseRedirect(request.META['PATH_INFO'])
    else:
        context['error'] = u"Каталог {} не существует".format(full_path)

    return django_render_to_response('admin/upload_schedule_file.html', context, RequestContext(request))


@login_required
@clean_temp
def upload_railway_file(request):
    full_path = os.path.normpath(os.path.join(settings.DATA_PATH, 'upload', 'railway'))

    if not os.path.exists(full_path):
        os.makedirs(full_path)

    context = {
        'full_path': full_path,
        'files': get_files_info(full_path)
    }

    if not request.FILES:
        return django_render_to_response('admin/upload_railway_file.html', context, RequestContext(request))

    try:
        fp = request.FILES['file']
        filepath = save_file_in_temp(fp)
        unzip_and_rewrite_railway_data(filepath, full_path)
        messages.add_message(request, messages.SUCCESS, mark_safe(u'Загрузили новый слой железных дорог'))
    except Exception as e:
        messages.add_message(request, messages.ERROR,
                             mark_for_escaping(u'При загрузке файла произошла ошибка. {}'.format(unicode(e))))

    return HttpResponseRedirect(request.META['PATH_INFO'])


def get_files_info(full_path):
    if not os.path.exists(full_path):
        return []

    files_info = []
    for filename in os.listdir(full_path):
        if filename.startswith('.'):
            continue

        filepath = os.path.join(full_path, filename)
        if os.path.isdir(filepath):
            continue

        stats = os.stat(filepath)
        files_info.append({
            'name': filename,
            'size': u'{}'.format(stats.st_size),
            'dt': datetime.fromtimestamp(stats.st_mtime).strftime('%Y-%m-%d %H:%M')
        })

    return files_info


def save_file_in_temp(fp):
    filepath = get_tmp_filepath('railway.zip')
    with open(filepath, 'wb+') as destination:
        for chunk in fp.chunks():
            destination.write(chunk)

    return filepath


def unzip_and_rewrite_railway_data(filepath, full_path):
    with open(filepath, 'rb') as fp:
        file_map = unpack_zip_file(fp)

    mif, mid = get_mif_and_mid_files(file_map)

    # remove old files (только если нашлись нужные файлы)
    for root, __, files in os.walk(full_path):
        for name in files:
            os.remove(os.path.join(root, name))

    shutil.move(mif, os.path.join(full_path, 'rw_full.mif'))
    shutil.move(mid, os.path.join(full_path, 'rw_full.mid'))


class FindMifAndMidError(SimpleUnicodeException):
    pass


def get_mif_and_mid_files(file_map):
    mif_candidates = [filename for filename in file_map if filename.endswith('.mif')]
    mid_candidates = [filename for filename in file_map if filename.endswith('.mid')]

    if len(mif_candidates) != 1 or len(mid_candidates) != 1:
        raise FindMifAndMidError(u'В загружаемом архиве должен быть один mif файл и один mid файл.')

    return file_map[mif_candidates[0]], file_map[mid_candidates[0]]


@login_required
def sindbad_stats(request):
    today = date.today()

    context = {'month': today.month,
               'year': today.year}

    if 'month' in request.GET:
        context['month'] = request.GET['month']

    context['month_int'] = int(context['month'])
    context['year_int'] = int(context['year'])

    if 'year' in request.GET:
        context['year'] = request.GET['year']

    context['years'] = range(2011, today.year + 1)
    context['months'] = [[i + 1, m[1]] for i, m in enumerate(russian_dt.MONTH_NAMES)]
    context['month_name'] = russian_dt.MONTH_NAMES[int(context['month']) -1][0]

    partner = Partner.objects.get(code='sindbad')

    entries = StatisticsEntry.for_month(partner, int(context['year']), int(context['month'])).\
                              values('day').\
                              annotate(orders=Count('id'), sum_price=Sum('price'))

    context['entries'] = list(entries)
    context['total_orders'] = sum([e['orders'] for e in entries])
    context['total_sum'] = sum([e['sum_price'] for e in entries])

    return django_render_to_response('admin/sindbad_stats.html', context, RequestContext(request))


@login_required
def download_db_file(request, app_label, model_name, pk, field_name):
    model = apps.get_model(app_label, model_name)
    if model and request.user.has_perm(u"%s.change_%s" % (app_label,
                                                          model.__name__.lower())):
        instance = model.objects.get(id=pk)
        memory_file = getattr(instance, field_name)
        response = HttpResponse(memory_file, content_type=memory_file.content_type)

        response['Content-Disposition'] = 'attachment; filename=%s' % str(Header(memory_file.name, 'utf-8'))

        return response
    else:
        HttpResponseRedirect("/admin/")


@login_required
def delete_db_file(request, app_label, model_name, pk, field_name):
    permission = (u"%s.change_%s" % (app_label, model_name)).lower()
    if not request.user.has_perm(permission):
        raise PermissionDenied()

    model = apps.get_model(app_label, model_name)

    instance = model.objects.get(id=pk)
    setattr(instance, field_name, None)
    instance.save()

    return HttpResponseRedirect(
        reverse("admin:%s_%s_change" % (app_label, model_name.lower()),
                args=(instance.id,))
    )


def load_trainpseudostationmap__file(fileobj):
    log.info(u"Загружаем файл %s", fileobj.name)

    reader = UnicodeDictReader(fileobj, ('number', 'pseudo_station_id', 'station_id'),
                               encoding='cp1251', delimiter=';', restkey='rest',
                               strip_values=True)
    for row in reader:
        if not row['number']:
            continue

        log.info(u"Загружаем замену для псевдостанций %s %s %s, %s",
                 row['number'], row['pseudo_station_id'], row['station_id'],
                 u" # ".join(map(unicode, row['rest'] or [])))

        if not TrainPseudoStationMap.objects.filter(number=row['number'], station=row['station_id']):
            try:
                tps_map = TrainPseudoStationMap()
                tps_map.number = row['number']
                tps_map.pseudo_station_id = row['pseudo_station_id']
                tps_map.station_id = row['station_id']
                tps_map.save()
                log.info(u"Замена загружена успешно %s %s %s", tps_map.number, tps_map.pseudo_station.title,
                         tps_map.station.title)
            except Exception:
                log.exception(u"Ошибка создания замены")
        else:
            log.warning(u"Такая замена уже есть в базе")


class TicketsSearchForm(forms.Form):
    adults = forms.IntegerField(required=False)
    children = forms.IntegerField(required=False)
    infants = forms.IntegerField(required=False)
    fromId = forms.CharField()
    toId = forms.CharField()
    when = forms.CharField()
    return_date = forms.CharField(required=False)
    klass = forms.CharField(required=False)

    def clean(self):
        data = self.cleaned_data

        from_id = data['fromId']

        try:
            data['point_from'] = Point.get_by_key(from_id)

        except Exception:
            self._errors['fromId'] = self.error_class([format_exc()])

        to_id = data['toId']

        try:
            data['point_to'] = Point.get_by_key(to_id)

        except Exception:
            self._errors['toId'] = self.error_class([format_exc()])

        today = date.today()

        def parse_date(text):
            try:
                d = dateparser.parse(
                    text,
                    today,
                    timedelta(settings.DAYS_TO_PAST)
                )

            except ValueError:
                raise forms.ValidationError(u'Wrong date format')

            return fix_date(d, today)

        data['date_forward'] = parse_date(data['when'])

        if data['return_date']:
            data['date_backward'] = parse_date(data['return_date'])

        return data

    def clean_klass(self):
        klasses = ['economy', 'business', 'first']

        klass = self.cleaned_data.get('klass')

        return klass if klass in klasses else 'economy'


@login_required
def get_configuration(request):
    database_settings = settings.DATABASES['default']

    data = {
        'caches': settings.CACHES.keys(),
        'replica_info': {
            'master': database_settings['HOST'],
            'replicas': database_settings.get('REPLICAS')
        },
    }

    json_data = json.dumps(
        data,
        default=serialize,
        ensure_ascii=False,
        separators=(',', ':'),
        indent=4
    )

    return HttpResponse(json_data, content_type="application/json; charset=utf-8")


@login_required
def get_settings(request):
    data = {k: getattr(settings, k) for k in dir(settings) if not k.startswith('_')}

    for k in data:
        try:
            json.dumps(data[k], default=serialize, ensure_ascii=False)
        except Exception:
            data[k] = unicode(data[k])

    json_data = json.dumps(
        data,
        default=serialize,
        ensure_ascii=False,
        separators=(',', ':'),
        indent=4,
    )

    return HttpResponse(json_data, content_type="application/json; charset=utf-8")


@login_required
def gtfs_form(request):
    from travel.avia.admin.lib.gtfs import convert_file

    if request.method == "POST":
        country = Country.objects.get(pk=request.POST.get('country').strip())
        file = request.FILES['file']
        xml = convert_file(file, country.code)
        response = HttpResponse(content_type='application/zip')
        response['Content-Disposition'] = 'filename=%s.zip' % datetime.now().strftime('%Y-%m-%d-%H-%M-%S')
        response.write(xml)
        return response
    else:
        widget = ForeignKeyRawIdWidget(Region._meta.get_field("country").rel, site)
        params = {'country_raw_widget': widget.render('country', None, attrs={'id': 'id_country'})}
        return django_render_to_response('admin/gtfsconverter/form.html', params, RequestContext(request))
