# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

import yt.wrapper as yt
from django.conf import settings
from django.core.exceptions import ObjectDoesNotExist
from django.http import HttpResponse
from django.utils.html import escape
from django.views.generic import View

from travel.avia.admin.avia_scripts.import_balance_partner_stat import (
    format_destination_path, add_task, get_base_path
)
from travel.avia.admin.lib.auth import AuthError
from travel.avia.admin.lib.exceptions import InternalError
from travel.avia.admin.lib.feature_flags import extended_report_flag_by_partner_code
from travel.avia.admin.lib.partner_authorization import check_access_to_report
from travel.avia.admin.lib.yt_helpers import configure_wrapper
from travel.avia.library.python.common.models.partner import Partner

MAX_DAYS = 31
TRY_GET_START_DATE = datetime(2019, 4, 22)


class PrepareReport(View):
    def get(self, request, *args, **kwargs):
        billing_client_id = int(request.GET.get('billing_client_id'))
        add_task(request.GET.get('left_date'), request.GET.get('right_date'), billing_client_id)
        return HttpResponse('Ok')


class BaseReport(View):
    COLUMNS = (
        'date',  # дата совершения клика по МСК часовому поясу
        'time',  # время совершения клика по МСК часовому поясу
        'from_IATA',  # код города вылета, использованный при тарификации перелета.
        'to_IATA',  # код города прилета, использованный при тарификации перелета.
        'from_Sirena',  # код города вылета, использованный при тарификации перелета, если у него нет IATA-кода
        'to_Sirena',  # код города прилета, использованный при тарификации перелета, если у него нет IATA-кода
        # название города вылета на английском, использованный при тарификации перелета. Фоллбек на название города на русском.
        'from_city',
        # название города прилета на английском, использованный при тарификации перелета. Фоллбек на название города на русском.
        'to_city',
        'month',  # месяц вылета
        'OW/RT',  # OW если вылет был в одном направлении, RT если туда и обратно
        # 'passengers',
        'adult_seats',
        'children_seats',
        'infant_seats',
        'price',  # стоимость клика из поля price
        'marker',  # значение поля marker
        'national_version',
    )

    EXTRA_COLUMNS = (
        'sale',  # был ли продан билет на стороне партнера
        'offer_price',  # цена билета
        'flight',  # рейс
        'source',  # источник (Советник, Сервис, Колдунщик, ...)
        'platform',  # платформа (Десктоп, тач, ...)
        'default_price',  # Цена из прайс-листа до введение ценообразования по источникам
    )

    BASE_PATH = get_base_path(settings.ENVIRONMENT)

    def _download_url(self, left_date, right_date, billing_client_id):
        url = (
            'https://hahn.yt.yandex-team.ru/api/v3/read_table?'
            'output_format[$value]=schemaful_dsv'
            '&output_format[$attributes][missing_value_mode]=print_sentinel'
            '&output_format[$attributes][enable_column_names_header]=true'
            '&output_format[$attributes][field_separator]=,'
            '&dump_error_into_response=true'
        )
        for column in self.COLUMNS:
            url += '&output_format[$attributes][columns][]=%s' % column

        if extended_report_flag_by_partner_code(Partner.objects.get(billing_client_id=billing_client_id).code):
            for column in self.EXTRA_COLUMNS:
                url += '&output_format[$attributes][columns][]=%s' % column

        url += '&path=%s' % format_destination_path(self.BASE_PATH, billing_client_id, left_date, right_date)

        return url


class Report(BaseReport):
    def get(self, request, *args, **kwargs):
        left_date = request.GET.get('left_date')
        right_date = request.GET.get('right_date')
        billing_client_id = int(request.GET.get('billing_client_id'))

        try:
            Partner.objects.get(billing_client_id=billing_client_id)
        except ObjectDoesNotExist:
            return HttpResponse(
                'Partner not found for billing_client_id={}'.format(billing_client_id),
                status=404,
            )

        configure_wrapper(yt)
        destination_table = format_destination_path(
            self.BASE_PATH, billing_client_id, left_date, right_date
        )
        if yt.exists(destination_table):
            return HttpResponse(self._download_url(escape(left_date), escape(right_date), billing_client_id))
        else:
            return HttpResponse('Table is not ready')


class TryGetReport(BaseReport):
    def get(self, request, *args, **kwargs):
        left_date = request.GET.get('left_date')
        right_date = request.GET.get('right_date')
        billing_client_id = request.GET.get('billing_client_id')

        for date_param, name in ((left_date, 'left_date'), (right_date, 'right_date')):
            if date_param is None:
                return HttpResponse('{} should be set'.format(name), status=400)

        escaped_left_date = escape(left_date)
        try:
            parsed_left_date = datetime.strptime(left_date, '%Y-%m-%d')
        except ValueError:
            return HttpResponse('Bad date format: {}'.format(escaped_left_date), status=400)

        escaped_right_date = escape(right_date)
        try:
            parsed_right_date = datetime.strptime(right_date, '%Y-%m-%d')
        except ValueError:
            return HttpResponse('Bad date format: {}'.format(escaped_right_date), status=400)

        if parsed_left_date > parsed_right_date:
            return HttpResponse('left_date should be less or equal to right_date', status=400)

        if parsed_left_date < TRY_GET_START_DATE:
            return HttpResponse(
                'left_date should not be before {}'.format(TRY_GET_START_DATE.strftime('%Y-%m-%d')),
                status=400,
            )

        if parsed_right_date - parsed_left_date > timedelta(days=MAX_DAYS):
            return HttpResponse('Number of days should be less or equal to {}'.format(MAX_DAYS), status=400)

        if billing_client_id is None:
            return HttpResponse('billing_client_id should be set', status=400)

        try:
            try:
                _billing_client_id = int(billing_client_id)
            except ValueError:
                return HttpResponse('Bad billing_client_id={}'.format(escape(billing_client_id)), status=400)

            user_ip = request.META.get('REMOTE_ADDR', None) or request.META.get('HTTP_X_REAL_IP', None)
            if (
                user_ip is None
                or not check_access_to_report(_billing_client_id, request.META.get('HTTP_AUTHORIZATION'), user_ip)
            ):
                return HttpResponse('Access denied', status=403)

        except AuthError as e:
            return HttpResponse('Access denied: {}'.format(e.msg), status=403)
        except Partner.DoesNotExist:
            return HttpResponse('Partner with billing_client_id={} is not found'.format(escape(billing_client_id)))
        except InternalError:
            return HttpResponse('Internal error', status=500)

        configure_wrapper(yt)
        destination_table = format_destination_path(
            self.BASE_PATH, billing_client_id, left_date, right_date
        )

        if yt.exists(destination_table):
            return HttpResponse(self._download_url(escaped_left_date, escaped_right_date, escape(billing_client_id)))
        else:
            add_task(left_date, right_date, int(billing_client_id))
            return HttpResponse('Table is not ready', status=202)
