# -*- coding: utf-8 -*-
import logging
from datetime import date, timedelta

from django.core.management import BaseCommand
from django.conf import settings

import yt.logger as ytlogger

from ._loggers import setup_logger
from ._yt_build_good_prices import (
    update_price_storage,
    update_shows,
    take_popular,
    update_quantiles,
    YT_PRICE_STORAGE_TABLE,
    YT_POPULAR_PRICE_STORAGE_TABLE,
    YT_QUANTILES_POPULAR_TABLE,
)


LOGGER_FMT_STR = '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
formatter = logging.Formatter(LOGGER_FMT_STR)
logging.basicConfig(format=LOGGER_FMT_STR)
log = logging.getLogger(__name__)


ALLOWED_ENVS = ['dev', 'production']


class Command(BaseCommand):
    help = 'Set file-flag for ping views for <duration> seconds'

    def add_arguments(self, parser):
        parser.add_argument(
            '--stdout', action='store_true', default=False,
            dest='add_stdout_handler', help='Add stdout handler',
        )

        parser.add_argument(
            '-d', '--days', type=int, default=1,
            dest='days', help='number of days to process',
        )

        parser.add_argument(
            '--good-services', required=True, type=lambda s: [service.strip() for service in s.split(',')],
            help='specify good services for calculation separated by comma, example: avia-travel,mavia-travel',
        )

    def handle(self, *db_names, **options):
        setup_logger(
            log,
            options.get('verbosity'),
            options.get('add_stdout_handler'),
            formatter=logging.Formatter(LOGGER_FMT_STR),
        )

        setup_logger(
            ytlogger.LOGGER,
            options.get('verbosity'),
            options.get('add_stdout_handler'),
            formatter=logging.Formatter(LOGGER_FMT_STR),
        )

        try:
            current_env = settings.ENVIRONMENT
            if current_env in ALLOWED_ENVS:
                start_date = date.today() - timedelta(days=options.get('days'))
                end_date = date.today() - timedelta(days=1)  # Yesterday
                log.info('Update shows')
                update_shows(start_date, end_date)

                log.info('Updating price storage: %s', YT_PRICE_STORAGE_TABLE)
                good_services = options['good_services']
                log.info('Good services %s', good_services)
                update_price_storage(start_date, end_date, good_services, YT_PRICE_STORAGE_TABLE)

                log.info('Taking popular flights')
                take_popular(
                    YT_PRICE_STORAGE_TABLE,
                    YT_POPULAR_PRICE_STORAGE_TABLE,
                )

                log.info('Calculating quatiles')
                update_quantiles(
                    YT_POPULAR_PRICE_STORAGE_TABLE,
                    YT_QUANTILES_POPULAR_TABLE
                )
            else:
                log.info("Skip YT part in %s environment", current_env)

        except Exception:
            log.exception('Exception in build_good_prices')
            raise
