# -*- coding: utf-8 -*-
from logging import getLogger
import json
import os
import csv

from django.core.management.base import BaseCommand

from travel.avia.library.python.common.models.geo import Station, StationCode, Settlement
from travel.avia.library.python.common.models.transport import TransportType


log = getLogger(__name__)
__location__ = os.path.realpath(
    os.path.join(os.getcwd(), os.path.dirname(__file__)))


class Command(BaseCommand):
    help = 'TASK: https://st.yandex-team.ru/RASPTICKETS-6647; IATA_API: iatacodes.org'

    def read_api_response(self, file):
        with open(os.path.join(__location__, '..', 'data', file)) as data_file:
            data = json.load(data_file)

        return {geo['code']: geo for geo in data['response']}

    def get_iatacodes_api_airports(self):
        return self.read_api_response('airports.json')

    def get_iatacodes_api_cities(self):
        return self.read_api_response('cities.json')

    def create_row(self, iata, avia_id, avia_title, api_title):
        avia_title = avia_title or ''
        api_title = api_title or ''

        status = '=' if avia_title.lower() == api_title.lower() else '!='

        return [iata, str(avia_id), avia_title, api_title, status]

    def get_airport_compare_table(self):
        iata_to_api_airports = self.get_iatacodes_api_airports()

        airports = list(Station.objects.filter(t_type_id=TransportType.PLANE_ID))
        id_to_airport = {s.id: s for s in airports}
        IATA_SYSTEM_ID = 4
        codes = StationCode.objects.filter(system_id=IATA_SYSTEM_ID,
                                           station_id__in=[s.id for s in airports])
        avia_iata_to_airport = {code.code: id_to_airport[code.station_id] for code in codes}

        avia_iatas = set(avia_iata_to_airport.keys())
        api_iatas = set(iata_to_api_airports.keys())

        iatas = avia_iatas.intersection(api_iatas)

        for iata in iatas:
            api_airport = iata_to_api_airports[iata]
            api_airport_title = api_airport['name']

            avia_airport = avia_iata_to_airport[iata]
            avia_airport_title = avia_airport.title_en

            yield self.create_row(
                iata=iata,
                avia_id=avia_airport.id,
                avia_title=avia_airport_title or '',
                api_title=api_airport_title
            )

        for iata in avia_iatas.difference(api_iatas):
            avia_airport = avia_iata_to_airport[iata]
            avia_airport_title = avia_airport.title_en

            yield self.create_row(
                iata=iata,
                avia_id=avia_airport.id,
                avia_title=avia_airport_title,
                api_title='UNKNOWN_IN_API'
            )

        for iata in api_iatas.difference(avia_iatas):
            api_airport = iata_to_api_airports[iata]
            api_airport_title = api_airport['name']

            yield self.create_row(
                iata=iata,
                avia_id='UNKNOWN_IN_AVIA',
                avia_title='UNKNOWN_IN_AVIA',
                api_title=api_airport_title
            )

    def get_city_compare_table(self):
        iata_to_api_city = self.get_iatacodes_api_cities()

        cities = list(Settlement.objects.filter(iata__isnull=False))
        avia_iata_to_city = {city.iata: city for city in cities}

        avia_iatas = set(avia_iata_to_city.keys())
        api_iatas = set(iata_to_api_city.keys())

        iatas = avia_iatas.intersection(api_iatas)

        for iata in iatas:
            api_city = iata_to_api_city[iata]
            api_city_title = api_city['name']

            avia_city = avia_iata_to_city[iata]
            avia_city_title = avia_city.title_en

            yield self.create_row(
                iata=iata,
                avia_id=avia_city.id,
                avia_title=avia_city_title or 'XXX',
                api_title=api_city_title
            )

        for iata in avia_iatas.difference(api_iatas):
            avia_city = avia_iata_to_city[iata]
            avia_city_title = avia_city.title_en

            yield self.create_row(
                iata=iata,
                avia_id=avia_city.id,
                avia_title=avia_city_title,
                api_title='UNKNOWN_IN_API'
            )

        for iata in api_iatas.difference(avia_iatas):
            api_city = iata_to_api_city[iata]
            api_city_title = api_city['name']

            yield self.create_row(
                iata=iata,
                avia_id='UNKNOWN_IN_AVIA',
                avia_title='UNKNOWN_IN_AVIA',
                api_title=api_city_title
            )

    def handle(self, *args, **options):
        with open(os.path.join(__location__, '..', 'data', 'compare-airports.csv'), 'w+') as csv_file:
            wr = csv.writer(csv_file, quoting=csv.QUOTE_ALL)
            wr.writerow(["iata", "avia id", "avia title", "api title"])
            for row in self.get_airport_compare_table():
                wr.writerow([z.encode("utf-8") for z in row])

        with open(os.path.join(__location__, '..', 'data', 'compare-city.csv'), 'w+') as csv_file:
            wr = csv.writer(csv_file, quoting=csv.QUOTE_ALL)
            wr.writerow(["iata", "avia id", "avia title", "api title"])
            for row in self.get_city_compare_table():
                wr.writerow([z.encode("utf-8") for z in row])
