# -*- coding: utf-8 -*-
"""Delete ticket-daemon cached results from SAAS by regexp
KEY FORMATS
    results: "/c66_c2_2018-04-28_2018-05-02_economy_1_0_0_ru/ru/aviakass"
    redirect_data: "//yandex/ticket-daemon/c2_c213_2018-05-12_None_economy_1_0_0_ru/ru/booktripruag_redirect_data"

EXAMPLES:
"./manage.py delete_from_saas -r c2_c213 -d 1": delete last day results and redirect_data on direction LED-MOW
"./manage.py delete_from_saas -r c2_c213.*ozon -d 1": delete last day results and redirect_data on direction LED-MOW for partner ozon
"./manage.py delete_from_saas -r c2_c213.*ozon_redirect_data -d 1": delete last day redirect_data on direction LED-MOW for partner ozon
"./manage.py delete_from_saas -r c2_c213.*ozon$ -d 1": delete last day results on direction LED-MOW for partner ozon

"""
import logging
import re
import time
from functools import partial

from django.core.management.base import BaseCommand
import yt.wrapper as yt

from travel.avia.admin.lib import yt_helpers as yth


log = logging.getLogger(__name__)

YT_LOG_PATH = '//home/rasp/logs/avia-saas-td-index-log'
UNIXTIME = time.time()


def _init_saas():
    from django.conf import settings
    from travel.avia.library.python.common.saas.index import LoggingIndex, SaasIndex

    SAAS_INDEXES = {}
    SAAS_SERVICES = (
        ('result', 'd01e5f46792489845c061a61a219123b'),
        ('redirect_data', '6fe68845e6a7d7b1d1858ec90e49050b'),
    )

    for saas_service, key in SAAS_SERVICES:
        SAAS_INDEXES[saas_service] = LoggingIndex(SaasIndex(
            index_host=settings.SAAS_INDEX_HOST, service_key=key,
        ))

    def _saas_index(key):
        if 'redirect_data' in key:
            return SAAS_INDEXES['redirect_data']
        else:
            return SAAS_INDEXES['result']

    return _saas_index


def map_expired(regexp, record):
    key = record['key']
    if UNIXTIME < record['expires_at'] and re.search(regexp, key):
        yield {'key': key}


class Command(BaseCommand):
    help = __doc__

    def add_arguments(self, parser):
        parser.add_argument(
            '-r', '--regexp',
            action='store',
            type=str,
            default='.*',
            help='regexp on deletion'
        )
        parser.add_argument(
            '-d', '--days', type=int, default=30,
            dest='days', help='number of days to scan logs',
        )

    def handle(self, *args, **options):
        yth.configure_wrapper(yt)
        source_tables = yth.last_logs_tables(yt, YT_LOG_PATH, options['days'])
        temp_table = yt.create_temp_table()
        yt.run_map(
            partial(map_expired, options['regexp']),
            source_table=source_tables,
            destination_table=temp_table,
        )
        _saas_index = _init_saas()
        for record in yt.read_table(temp_table, raw=False):
            key = record['key']
            try:
                log.info('DELETING %s', key)
                _saas_index(key).delete(key)
            except Exception as exc:
                log.exception('Unexpected SaaS error: %r', exc)
