# -*- coding: utf-8 -*-
from collections import defaultdict
from datetime import date, timedelta
from logging import getLogger
import time

from django.core.management.base import BaseCommand
import requests

from travel.avia.library.python.common.models.geo import Station, StationCode
from travel.avia.library.python.common.models.transport import TransportType


log = getLogger(__name__)
TIMEOUT_TIME = 0.1


def get_livestore_request_data(country, departure_iata, arrival_iatas, date_from, date_to):
    arrival_iatas = set(arrival_iatas)
    arrival_iatas.discard(departure_iata)
    arrival_iatas = list(arrival_iatas)
    if not arrival_iatas:
        log.warn('Can\'t build livestore url for %s', departure_iata)
        return None

    url = ('https://yandexapi.dohop.com/api/v1/livestore'
           '/en/{}/per-airport/{}/{}/{}/{}'.format(country,
                                                   departure_iata,
                                                   ','.join(arrival_iatas),
                                                   date_from.isoformat(),
                                                   date_to.isoformat()))

    return {
        'url': url,
        'departure_iata': departure_iata
    }


def get_borders(today):
    return (today + timedelta(1)), (today + timedelta(46))


def find_trash_codes(codes, now):
    """
    Дохом не знает часть наших IATA, так как справочника нет.
    Поэтому попробуем восстановить 'справочник' по их 'ошибкам'

    Будем слать обычный запрос к дохопу. Если дохоп ответит нормально,
    то все коды нормальные, иначе пытаемся распарсить и извлечь проблемный код.

    Так как апи GET, то дохопу нельзя отослать все коды сразу, поэтому
    шлем коды пачками по PACK_SIZE штук
    """
    log.info('The search for trash codes has started:')
    date_from, date_to = get_borders(now)
    trash_codes = set()

    PACK_SIZE = 10

    for i in range(0, len(codes), PACK_SIZE):
        log.info('Pack Number: %d', i/PACK_SIZE)
        pack = set(codes[i: (i+PACK_SIZE)])

        while len(pack):
            arrival_iatas = list(pack)
            log.info('Codes count: %d', len(arrival_iatas))

            url = get_livestore_request_data(country='RU',
                                             departure_iata='SIP',
                                             arrival_iatas=arrival_iatas,
                                             date_from=date_from,
                                             date_to=date_to)['url']
            log.info('Send request: %s', url)
            response = requests.get(url).json()
            time.sleep(TIMEOUT_TIME)

            error = response.get('error', None)
            if 'error' not in response:
                break

            if not error.startswith('unknown airport '):
                raise Exception(error)

            unknown_code = error[16:]
            trash_codes.add(unknown_code)
            pack.remove(unknown_code)

    log.info('The search for trash codes is done: %r', sorted(trash_codes))
    return trash_codes


class Command(BaseCommand):
    help = 'RASPTICKETS-6485. Find alive hidden airports.'

    def handle(self, *args, **options):
        now = date.today()
        log.info('Start download airports')
        airports = list(Station.objects.filter(hidden=True,
                                               t_type_id=TransportType.PLANE_ID,
                                               country_id__isnull=False))
        id_to_airport = {s.id: s for s in airports}
        log.info('Finish download airports. Count of airports: %d',
                 len(airports))
        log.info('Start pre-process data:')

        IATA_SYSTEM_ID = 4
        codes = StationCode.objects.filter(system_id=IATA_SYSTEM_ID,
                                           station_id__in=id_to_airport.keys())
        iata_to_airport = {code.code: id_to_airport[code.station_id] for code in codes}
        id_to_iata = {code.station_id: code.code for code in codes}

        trash_codes = find_trash_codes(iata_to_airport.keys(), now)
        for code in trash_codes:
            airport = iata_to_airport.pop(code)
            id_to_iata.pop(airport.id)

        airports_with_iata = iata_to_airport.values()

        contry_to_airports = defaultdict(list)
        for s in airports_with_iata:
            contry_to_airports[s.country_id].append(s)

        hidden_airports = [s for s in airports_with_iata if s.hidden]
        log.info('Start build livestore urls for %d airports',
                 len(hidden_airports))

        date_from, date_to = get_borders(now)
        livestore_data = []
        for airport in hidden_airports:
            iata = id_to_iata[airport.id]
            arrival_iatas = [id_to_iata[s.id] for s in contry_to_airports[airport.country_id]]

            for country in ['RU', 'IS', 'GB']:
                data = get_livestore_request_data(country=country,
                                                  departure_iata=iata,
                                                  arrival_iatas=arrival_iatas,
                                                  date_from=date_from,
                                                  date_to=date_to)

                if data:
                    livestore_data.append(data)
        log.info('Built requests for dohop. Requests count: %d',
                 len(livestore_data))

        unhidden_iatas = []
        unhidden_airports = []
        for i, data in enumerate(livestore_data):
            if i % 10 == 0:
                log.info('Process %d', i)

            is_unhidden = False
            try:
                r = requests.get(data['url']).json()
                time.sleep(TIMEOUT_TIME)

                if 'error' in r:
                    raise Exception(r['error'])

                is_unhidden = len(r['airports']) > 0
            except Exception, e:
                log.err("Request to dohop has failed: %r", e)

            if (is_unhidden):
                code = data['departure_iata']
                station_id = iata_to_airport[code]
                log.info('It\'s alive: %s %d', code, station_id)
                unhidden_iatas.append(code)
                unhidden_airports.append(station_id)

        log.info('unhidden_iata: %r', sorted(unhidden_iatas))
        log.info('unhidden_airports: %r', sorted(unhidden_airports))
        log.info('Finish!')
