# -*- coding: utf-8 -*-
import logging
from argparse import ArgumentParser

import os
from datetime import datetime

from django.core.management import BaseCommand
from django.conf import settings

import yt.logger as ytlogger
import yt.wrapper as ytw

from travel.avia.admin.lib import yt_helpers
from travel.avia.admin.lib.merge_tables import merge_table, TableMergeError
from travel.avia.admin.lib.logs import create_current_file_run_log
from ._loggers import setup_logger


LOGGER_FMT_STR = '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
formatter = logging.Formatter(LOGGER_FMT_STR)
logging.basicConfig(format=LOGGER_FMT_STR)
log = logging.getLogger(__name__)


ALLOWED_ENVS = ['production', 'dev']
LOGS_PATH_PREFIX = '//home/logfeller/logs/'
DESTINATION_PREFIX_FORMAT = '//home/{}/logs/'


def merge_logs_with_schema(src_path, dst_path, date_format,
                           min_date=None, max_date=None):
    if not ytw.exists(dst_path):
        ytw.create('map_node', dst_path, recursive=True)

    if not ytw.exists(src_path):
        log.info('Source path [%s] does not exist, abort', src_path)
        return

    for table in ytw.search(src_path, node_type='table'):
        table_name = table.split('/')[-1]

        table_date = datetime.strptime(table_name, date_format)
        if min_date is not None and table_name < min_date:
            continue

        if max_date is not None and table_name > max_date:
            continue

        dst_table = os.path.join(dst_path, table_date.strftime('%Y-%m-%d'))
        try:
            merge_table(table, dst_table)
        except TableMergeError as e:
            log.exception(e)
            continue


class Command(BaseCommand):
    help = 'Merge yt logs'

    def add_arguments(self, parser):
        # type: (ArgumentParser) -> None

        parser.add_argument(
            dest='log_name', help='name of the log, e.g. avia-suggests-log'
        )

        parser.add_argument(
            '--project', dest='project', choices={'avia', 'rasp'},
            help='Project name'
        )

        parser.add_argument(
            '--stdout', action='store_true', default=False,
            dest='add_stdout_handler', help='Add stdout handler',
        )

        parser.add_argument(
            '-m', '--mode', default='30min',
            dest='mode', help='1d, 30min or ultrafast'
        )

        parser.add_argument(
            '--min-date', default=None,
            dest='min_date', help='minimal date in YYYY-mm-ddTHH:MM:SS format. If empty, we will try to merge all available data'
        )

        parser.add_argument(
            '--max-date', default=None,
            dest='max_date', help='maximal date. Same as min-date'
        )

    def handle(self, *db_names, **options):
        create_current_file_run_log()
        for logger in (log, ytlogger.LOGGER):
            setup_logger(
                logger,
                verbosity=options.get('verbosity'),
                add_stdout_handler=options.get('add_stdout_handler'),
                formatter=logging.Formatter(LOGGER_FMT_STR),
            )

        try:
            current_env = settings.ENVIRONMENT
            if current_env not in ALLOWED_ENVS:
                log.info('Can\'t work in %s', current_env)
                return

            yt_helpers.configure_wrapper(ytw)

            log_path = os.path.join(LOGS_PATH_PREFIX, options['log_name'])

            if options.get('mode') == 'd':
                log_path = os.path.join(log_path, '1d')
                date_format = '%Y-%m-%d'
            elif options.get('mode') == '30min':
                log_path = os.path.join(log_path, '30min')
                date_format = '%Y-%m-%dT%H:%M:%S'
            elif options.get('mode') == 'ultrafast':
                log_path = os.path.join(log_path, 'stream', '5min')
                date_format = '%Y-%m-%dT%H:%M:%S'
            else:
                log.error('Unknown mode: %s', options.get('mode'))
                raise Exception

            merge_logs_with_schema(
                src_path=log_path,
                dst_path=os.path.join(DESTINATION_PREFIX_FORMAT.format(options['project']), options['log_name']),
                date_format=date_format,
                min_date=options.get('min_date'),
                max_date=options.get('max_date'),
            )

        except Exception:
            log.exception('ERROR')
