# -*- coding: utf-8 -*-
from __future__ import unicode_literals
import logging
import subprocess

from django.conf import settings
from django.core.management import BaseCommand
from django.utils.encoding import smart_unicode

from travel.avia.admin.lib.logs import create_current_file_run_log

from ._loggers import setup_logger_verbosity

log = logging.getLogger(__name__)


# ./manage.py rsync_import_rasp_media --config_name rasp-media-data
class Command(BaseCommand):
    def add_arguments(self, parser):
        parser.add_argument('-c', '--config_name', dest='config_name')

    def handle(self, **options):
        setup_logger_verbosity(log, options['verbosity'], self.stdout, self.stderr)
        create_current_file_run_log()
        try:
            _handle_rsync_import(options)
        except Exception:
            log.exception('Handle fail')


def _handle_rsync_import(options):
    config_name = options['config_name']
    log.info('Запускаем rsync—импорт для конфига %s', config_name)
    try:
        sources = settings.RSYNC_IMPORT_CONFIGS[config_name]
    except KeyError:
        log.error('Такого конфига для rsync нет: %r', config_name)
        return

    rsync(config_name, sources)


IO_TIMEOUT = 600
CONNECTION_TIMEOUT = 10


def rsync(config_name, sources):
    processes = []

    options = ['-avzL', '--delay-updates', '--delete-after', '--delete-excluded']
    options += ['--timeout={}'.format(IO_TIMEOUT), '--contimeout={}'.format(CONNECTION_TIMEOUT)]

    for source in sources:
        command = [settings.RSYNC] + options + ['rsync://%s/%s' % (source.host, source.share)]

        if source.exclude:
            for path in source.exclude:
                command.append('--exclude=' + path)

        args = command + [source.copy_to]
        log.info('rsync command: [%s]', ' '.join(args))

        process = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
        processes.append(process)

    has_errors = False
    has_notices = False

    # Ждем завершение процессов
    for process in processes:
        process.wait()

        stderr = process.stderr.read()
        stdout = process.stdout.read()

        if stderr:
            log.error('stderr:/n%s', smart_unicode(stderr))
            has_errors = True

        if stdout:
            log.info('stdout:/n%s', smart_unicode(stdout))
            has_notices = True

    if has_errors:
        log.error('Запуск rsync %s прошел с ошибками', config_name)
    elif has_notices:
        log.warning('Запуск rsync %s прошел с замечаниями', config_name)
    else:
        log.info('Запуск rsync %s прошел успешно', config_name)
