# coding: utf8
from __future__ import unicode_literals

import struct
import urllib2
from collections import Counter
from logging import getLogger

from django.core.management.base import BaseCommand

from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.image_processing import svg2image

LOGO_SIZE = 30
TRANSPARENT_COLOR = (255, 255, 255)

log = getLogger(__name__)


class Command(BaseCommand):
    help = 'Updates the colors of the companies on the basis of the color of their logo'

    def add_arguments(self, parser):
        parser.add_argument(
            '--force',
            action='store_true',
            default=False,
            help='Force update all colors'
        ),
        parser.add_argument(
            '--html-path',
            action='store',
            default=False,
            help='Path to render html with companies\' logos'
        )

    def handle(self, *args, **options):
        update_company_colors(options['force'], options['html_path'])


def update_company_colors(force, html_path):
    company_model_qs = build_company_qs(force, html_path)
    log.info('Need to calculate bgcolor for %s companies', company_model_qs.count())

    company_list = [get_company(c) for c in company_model_qs]

    update_colors(company_list, force)

    if html_path:
        render_html(company_list, html_path)


def build_company_qs(force, html_path):
    company_model_qs = Company.objects.exclude(sirena_id=None, iata=None).exclude(svg_logo2=None).exclude(svg_logo2='')
    if not force and not html_path:
        company_model_qs = company_model_qs.filter(logo_bgcolor='')
    return company_model_qs


def get_company(company_model):
    log.info('Working on %s', company_model)
    logo_url = company_model.svg_logo2.url
    logo_content = download_logo(logo_url)
    color = get_color(logo_content)

    return {
        'model': company_model,
        'logo_url': logo_url,
        'color': color
    }


def download_logo(url):
    return urllib2.urlopen(url).read()


def get_color(logo_content):
    image_data = get_image_data(logo_content)
    image_data = image_data.convert('RGBA')
    pixel = get_border_pixel(image_data)

    return rgb2hex(pixel)


def get_border_pixel(image_data):
    pixels = []

    x_max, y_max = image_data.size
    for x in (0, x_max - 1):
        for y in range(y_max):
            pixels.append(image_data.getpixel((x, y)))

    for y in (0, y_max - 1):
        for x in range(1, x_max - 1):
            pixels.append(image_data.getpixel((x, y)))

    return Counter(pixels).most_common(1)[0][0]


def get_image_data(logo_content):
    return svg2image(logo_content, size=(LOGO_SIZE, LOGO_SIZE))


def check_transparent(rgba):
    return len(rgba) == 4 and rgba[3] == 0


def rgb2hex(rgba):
    rgb = TRANSPARENT_COLOR if check_transparent(rgba) else (rgba[i] for i in range(3))
    return '#' + struct.pack('BBB', *rgb).encode('hex')


def update_colors(company_list, force_update):
    for company in company_list:
        if force_update or company['model'].logo_bgcolor == '':
            update_company_color(company)


def update_company_color(company):
    old_color = company['model'].logo_bgcolor
    if old_color.lower() != company['color'].lower():
        company['model'].logo_bgcolor = company['color']
        company['model'].save(update_fields=('logo_bgcolor',))
        log.info('The color of the company %s was changed from <%s> to <%s>', company['model'], old_color, company['color'])


def render_html(company_list, path):
    with open(path, 'w') as f:
        f.write("""
            <!DOCTYPE html>
            <html>
                <head>
                    <meta charset="utf-8">
                    <title>Logo SVGs</title>
                    <style>
                        body {{
                            display: flex;
                            flex-wrap: wrap;
                            background-color: #ece8dd;
                        }}
                        div {{
                            margin: 2px;
                            width: 100px;
                        }}
                        div > div {{
                            width: {logo_size}px;
                            height: {logo_size}px;
                            margin: 0;
                            background-size: contain;
                        }}
                    </style>
                </head>
                <body>
        """.format(logo_size=str(LOGO_SIZE)))

        for company in company_list:
            f.write("""
                <div style="background-color: {color}" data-id="{company_id}">
                    <div style="background-image: url({logo_url})"></div>
                </div>
            """.format(color=company['color'], company_id=company['model'].id, logo_url=company['logo_url']))

        f.write("""
                </body>
            </html>
        """)

        log.info('HTML file with logos: {html_path} created'.format(html_path=path))
