# -*- coding: utf-8 -*-
import re
import subprocess

from datetime import datetime, timedelta
from dateutil.relativedelta import relativedelta
from functools import update_wrapper

from django import forms
from django.conf import settings
from django.conf.urls import url
from django.contrib import messages, admin
from django.contrib.admin import widgets
from django.contrib.admin.filters import SimpleListFilter
from django.contrib.contenttypes.models import ContentType
from django.core.files.uploadhandler import TemporaryFileUploadHandler
from django.core.mail import send_mail
from django.db.models import Count
from django.forms.utils import flatatt
from django.http import HttpResponseRedirect
from django.db import models
from django.db.models import Q
from django.shortcuts import render
from django.template.response import TemplateResponse
from django.utils.http import urlencode
from django.views.decorators.csrf import csrf_exempt, csrf_protect
from django.utils.encoding import force_unicode, smart_unicode
from django.utils.decorators import method_decorator
from django.utils.html import escape, conditional_escape
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.avia_data.models import (
    AdminSettlementBigImage, AviaCompany, AviaDirection, AviaDirectionNational, AviaRecipe, AviaSettlement,
    AviaSettlementNational, AviaWizardDirection, AvgPrice, BalanceRedirect, BalanceRedirectPrepeared,
    BestOffers, CompanyRating, CompanyTariff, Currency, CurrencyLang, CurrencyTranslation, CustomCacheTimeDirection,
    FlightNumber, FlightRating, FlightReview, FlightReviewSource, MinCountryPrice, MinPrice, NationalVersion,
    NearCountries, NearDirection, PARTNER_REVIEW_RESULT_CHOICES, PartnerReview, RazladkiLastProcessedTable,
    Registry, SelfBookCompany, SelfBookDirection, SelfBookNationalVersion, SelfBookPartner, SelfBookRule,
    SeoDirection, ShowLog, SimilarDirection, TopDirection, TopFlight, TourDeparture, TourDestination, ToursMatrix,
)
from travel.avia.library.python.common.models.partner import Partner
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.models_utils.i18n import L_field

from travel.avia.admin.lib.admin_options import RaspExportModelAdmin

csrf_exempt_m = method_decorator(csrf_exempt)

RE_FLIGHT_NUMBER_IATA = re.compile(
    ur'(([A-Z][0-9A-Z])|([0-9A-Z][A-Z])) [0-9]{1,4}[A-Z]?'
)
RE_FLIGHT_NUMBER_SIRENA = re.compile(
    ur'(([А-ЯЁ][0-9А-ЯЁ])|([0-9А-ЯЁ][А-ЯЁ])) [0-9]{1,4}[А-ЯЁ]?'
)


class NullFilter(SimpleListFilter):
    def lookups(self, request, model_admin):
        return (
            ('1', _(u'Да'), ),
            ('0', _(u'Нет'), ),
        )

    def queryset(self, request, queryset):
        kwargs = {self.parameter_name: None}

        if self.value() == '0':
            return queryset.filter(**kwargs)
        if self.value() == '1':
            return queryset.exclude(**kwargs)

        return queryset


class SettlementNullFilter(NullFilter):
    title = _(u'Есть город')
    parameter_name = 'settlement'


class DistanceNullFilter(NullFilter):
    title = _(u'Отображать ближайшие')
    parameter_name = 'default_distance'


class CustomPartnerReviewResultFilter(SimpleListFilter):
    title = _(u'Результат')
    parameter_name = 'result__in'

    def lookups(self, request, model_admin):
        return list(PARTNER_REVIEW_RESULT_CHOICES) + [
            ('problem,error', _(u'Не прошла'))
        ]

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(result__in=self.value().split(','))
        else:
            return queryset


class CustomPartnerReviewPartnersFilter(SimpleListFilter):
    title = _(u'Партнеры')
    parameter_name = 'partner__id__exact'

    def lookups(self, request, model_admin):
        partner_ids = PartnerReview.objects.values_list('partner', flat=True).distinct()
        partners = Partner.objects.in_bulk(partner_ids).items()

        return partners

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(partner=self.value())

        else:
            return queryset


class VerboseManyToManyRawIdWidget(widgets.ManyToManyRawIdWidget):
    """
    A Widget for displaying ManyToMany ids in the "raw_id" interface rather than
    in a <select multiple> box.
    Display user-friendly value like the ForeignKeyRawId widget

    https://djangosnippets.org/snippets/2108/
    """
    def label_for_value(self, value):
        values = value.split(',')
        str_values = []
        key = self.rel.get_related_field().name
        for v in values:
            try:
                obj = self.rel.to._default_manager.using(self.db).get(**{key: v})
                # manage unicode error
                x = smart_unicode(obj)
                # no HTML
                str_values.append(escape(x))
            except self.rel.to.DoesNotExist:
                str_values.append(u'???')
        return u'&nbsp;<strong>%s</strong>' % (u',&nbsp;'.join(str_values))


class PreTagWidget(widgets.AdminTextareaWidget):
    def render(self, name, value, attrs=None):
        if value is None:
            value = ''
        final_attrs = self.build_attrs(attrs, name=name)
        return mark_safe(
            u'<pre%s>%s</pre>' % (
                flatatt(final_attrs),
                conditional_escape(force_unicode(value)),
            )
        )


class AviaRecipeForm(forms.ModelForm):
    class Meta:
        model = AviaRecipe
        exclude = []

    def _geo_id_clean(self, field):
        no_geoid = []
        field_data = self.cleaned_data.get(field)
        if field_data:
            no_geoid = [i for i in field_data if not i._geo_id]

        if no_geoid:
            text = _(u'Нет geoid: %s') % u', '.join([smart_unicode(i) for i in no_geoid])
            raise forms.ValidationError(text)

        return field_data

    def clean_countries(self):
        return self._geo_id_clean('countries')

    def clean_regions(self):
        return self._geo_id_clean('regions')

    def _missing_titles(self, national_version, langs):
        missing_titles = []

        for lang in langs:
            title = self.cleaned_data.get('title_%s' % lang)

            if not title:
                missing_titles.append((national_version, lang))

        return missing_titles

    def _clean_titles(self):
        missing_titles = []
        titles_required_map = {
            'ru': ['ru', 'en'],
            'ua': ['uk', 'ru', 'en'],
            'tr': ['tr', 'en'],
        }

        for national_version, langs in titles_required_map.items():
            enabled = self.cleaned_data.get('enabled_%s' % national_version)

            if enabled:
                missing_titles += self._missing_titles(national_version, langs)

        if missing_titles:
            messages = [u'%s для %s' % (n, l) for l, n in missing_titles]

            raise forms.ValidationError(_(u'Не заполнено название %s' % ', '.join(messages)))

    def clean(self):
        self._clean_titles()

        return self.cleaned_data


def copy_recipe(modeladmin, request, queryset):
    for model in queryset:
        model.pk = None
        model.save()


copy_recipe.short_description = _(u'Копировать')


class AviaRecipeAdmin(RaspExportModelAdmin):
    list_display = (
        'title_ru', 'recipe_type', 'date_start', 'date_end', 'backward_date_start', 'backward_date_end',
        'enabled_ru', 'enabled_ua', 'enabled_tr', 'enabled_com')
    raw_id_fields = ('countries', 'regions', 'settlements', 'from_regions', 'from_settlements')
    list_filter = ('enabled_ru', 'enabled_ua', 'enabled_tr', 'enabled_com')
    actions = [copy_recipe]

    fieldsets = (
        ('', {
            'fields': ('recipe_type',) + L_field.admin_fields(AviaRecipe, ['title']) + (
                'enabled_ru', 'enabled_ua', 'enabled_tr', 'enabled_com', 'enabled_kz', 'slug')
        }),
        (_(u'Куда'), {
            'fields': ('countries', 'regions', 'settlements')
        }),
        (_(u'Когда'), {
            'fields': ('week_days', 'date_start', 'date_end', 'backward_date_start', 'backward_date_end')
        }),
        (_(u'Откуда'), {
            'fields': ('from_regions', 'from_settlements', 'show_start', 'show_end', 'show_year')
        }),
        ('', {
            'fields': ('order',)
        }),
        ('SEO', {
            'fields': (L_field.admin_fields(AviaRecipe, ['h1']) +
                       ('meta_title_key',) +
                       L_field.admin_fields(AviaRecipe, ['meta_title']) +
                       L_field.admin_fields(AviaRecipe, ['meta_description']))
        }),
    )

    form = AviaRecipeForm

    def formfield_for_dbfield(self, db_field, **kwargs):
        if db_field.name in ('countries', 'regions', 'settlements', 'from_regions', 'from_settlements'):
            db = kwargs.get('using')
            kwargs.pop('request', None)
            kwargs['widget'] = VerboseManyToManyRawIdWidget(
                db_field.rel, self.admin_site, using=db
            )

            return db_field.formfield(**kwargs)

        return super(AviaRecipeAdmin, self).formfield_for_dbfield(db_field, **kwargs)

admin.site.register(AviaRecipe, AviaRecipeAdmin)


class ImageInput(forms.ClearableFileInput):
    def _get_image_thumbnail(self, url):
        return mark_safe('''
            <p><a href="{url}" target="_blank">
                <img style="max-width: 600px;" src="{url}">
            </a></p>
        '''.format(url=url))

    def render(self, name, value, attrs=None):
        html = ''

        if name == 'url2':
            html = super(ImageInput, self).render(name, value, attrs)

        if value and getattr(value, 'url', None):
            html += self._get_image_thumbnail(value.url)

        return html


class SettlementBigImageAdmin(RaspExportModelAdmin):
    list_display = ('settlement', 'url2_with_size',)
    raw_id_fields = ('settlement',)
    list_filter = (SettlementNullFilter,)

    formfield_overrides = {
        models.ImageField: {'widget': ImageInput}
    }

    def url2_with_size(self, obj):
        if obj.url2:
            link = '<a href="{url}">{url}</a>'.format(url=escape(obj.url2.url))
            return mark_safe(link)
        return ''
    url2_with_size.short_description = 'Ссылка на изображение в Аватарнице'

    # Всегда временно сохраняем файл, даже маленький
    # при добавлении и изменении, магию с csrf смотри тут:
    # https://docs.djangoproject.com/en/1.4/topics/http/file-uploads/#modifying-upload-handlers-on-the-fly

    @csrf_exempt_m
    def add_view(self, request, *args, **kwargs):
        request.upload_handlers = [TemporaryFileUploadHandler()]

        view = super(SettlementBigImageAdmin, self).add_view

        if request.method == 'POST':
            messages.add_message(request, messages.INFO, _(u"Создание миниатюр запущено в фоне"))

        return csrf_protect(view)(request, *args, **kwargs)

    @csrf_exempt_m
    def change_view(self, request, *args, **kwargs):
        request.upload_handlers = [TemporaryFileUploadHandler()]

        view = super(SettlementBigImageAdmin, self).change_view
        return csrf_protect(view)(request, *args, **kwargs)

admin.site.register(AdminSettlementBigImage, SettlementBigImageAdmin)


class DefaultDistanceForm(forms.Form):
    _selected_action = forms.CharField(widget=forms.MultipleHiddenInput)
    default_distance = forms.IntegerField()


class NearDirectionAdmin(RaspExportModelAdmin):
    list_display = ('departure_settlement', 'arrival_settlement',
                    'min_distance', 'default_distance', 'max_distance')

    fieldsets = (
        ('', {
            'fields': ('departure_settlement', 'arrival_settlement')
        }),
        (_(u'Дистанция'), {
            'fields': ('min_distance', 'default_distance', 'max_distance')
        }),
    )

    raw_id_fields = ('departure_settlement', 'arrival_settlement')

    search_fields = ('arrival_settlement__title',)
    list_filter = (DistanceNullFilter,)

    def get_urls(self):
        # Враппер для проверки пермишеннов
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        info = self.model._meta.app_label, self.model._meta.model_name

        urls = [url(r'^recalc/$', wrap(self.recalc), name='%s_%s_recalc' % info)]
        return urls + super(NearDirectionAdmin, self).get_urls()

    def recalc(self, request):
        messages.add_message(request, messages.INFO, _(u"Пересчёт аэропортов запущен"))

        subprocess.Popen(
            ["flock", "-n", "/tmp/avia-near-airports-recalc.lock", "-c",
             "python avia/scripts/airports_recalc.py"],
            cwd=settings.PROJECT_PATH
        )

        return HttpResponseRedirect('..')

    def send_email_notify(self, queryset, default_distance):
        receipents = ['rasp-process@yandex-team.ru']
        message_body = 'Set default distance to %s:\n\n' % default_distance

        for item in sorted(queryset):
            message_body += '%s\n' % item

        send_mail(
            'Default distance was changed',
            message_body,
            settings.SERVER_EMAIL,
            receipents,
            fail_silently=False
        )

    def reset_default_distance(modeladmin, request, queryset):
        queryset.update(default_distance=None)
        modeladmin.send_email_notify(queryset, None)

    def set_default_distance(modeladmin, request, queryset):
        form = None

        if 'apply' in request.POST:
            form = DefaultDistanceForm(request.POST)

            if form.is_valid():
                default_distance = form.cleaned_data['default_distance']
                queryset.update(default_distance=default_distance)
                modeladmin.send_email_notify(queryset, default_distance)
                modeladmin.message_user(request, "Successfully updated: %s" % len(queryset))

                return HttpResponseRedirect(request.get_full_path())

        if not form:
            form = DefaultDistanceForm(initial={
                '_selected_action': request.POST.getlist(admin.ACTION_CHECKBOX_NAME)
            })

        return render(
            request,
            'avia_data/set_default_distance.html',
            {'items': queryset, 'form': form, 'title': u'Изменение дистаниции'}
        )

    actions = [reset_default_distance, set_default_distance]
    set_default_distance.short_description = u"Установить дистанцию по-умолчанию"
    reset_default_distance.short_description = u"Принудительно скрыть блок аэропортов"


admin.site.register(NearDirection, NearDirectionAdmin)


class AviaDirectionAdmin(RaspExportModelAdmin):
    list_display = ('departure_settlement', 'arrival_settlement',
                    'direct_flights', 'connecting_flights', 'popularity')
    raw_id_fields = ('arrival_settlement', 'departure_settlement')
    search_fields = ('departure_settlement__title',)

admin.site.register(AviaDirection, AviaDirectionAdmin)


class AviaDirectionNationalAdmin(RaspExportModelAdmin):
    list_display = ('departure_settlement', 'arrival_settlement',
                    'direct_flights', 'connecting_flights',
                    'popularity', 'national_version')
    raw_id_fields = ('arrival_settlement', 'departure_settlement')
    search_fields = ('departure_settlement__title',)
    list_filter = ('national_version',)

admin.site.register(AviaDirectionNational, AviaDirectionNationalAdmin)


class AviaSettlementAdmin(RaspExportModelAdmin):
    list_display = ('settlement', 'arrival', 'popularity')
    raw_id_fields = ('settlement',)
    search_fields = ('settlement__title',)
    list_filter = ('arrival',)

admin.site.register(AviaSettlement, AviaSettlementAdmin)


class AviaSettlementNationalAdmin(RaspExportModelAdmin):
    list_display = ('settlement', 'arrival', 'popularity', 'national_version')
    raw_id_fields = ('settlement',)
    search_fields = ('settlement__title',)
    list_filter = ('arrival', 'national_version')

admin.site.register(AviaSettlementNational, AviaSettlementNationalAdmin)


class FlightReviewSourceAdmin(RaspExportModelAdmin):
    list_display = ('code', 'enabled', )
    list_filter = ('enabled',)


admin.site.register(FlightReviewSource, FlightReviewSourceAdmin)


class FlightNumberAdmin(RaspExportModelAdmin):
    list_display = ('id', 'flight_number', )

admin.site.register(FlightNumber, FlightNumberAdmin)


class FlightNumbersInline(admin.TabularInline):
    model = FlightReview.flight_numbers.through

    can_delete = False
    extra = 0


class FlightReviewForm(forms.ModelForm):
    def clean_raw_flight_numbers(self):
        raw_flight_numbers = filter(
            None,
            [n.strip().upper() for n in re.split('[\n;,]+', self.cleaned_data['raw_flight_numbers'])]
        )

        bad_flight_numbers = set()
        bad_airline_codes = set()
        prepared_flight_numbers = []

        if raw_flight_numbers:
            for n in raw_flight_numbers:
                match = RE_FLIGHT_NUMBER_IATA.match(n) or \
                    RE_FLIGHT_NUMBER_SIRENA.match(n)

                if not match or match.group() != n:
                    bad_flight_numbers.add(n)
                    continue

                airline_code, _ = n.split(' ')
                if airline_code == 'FV':
                    try:
                        airline = Company.objects.get(
                            t_type_id=TransportType.PLANE_ID, title="Россия"
                        )
                    except Company.DoesNotExist:
                        airline = None
                else:
                    airline = Company.objects.filter(
                        Q(iata=airline_code) | Q(sirena_id=airline_code)
                    ).order_by(
                        '-priority'
                    ).first()

                if airline:
                    prepared_flight_numbers.append([airline, n])
                else:
                    bad_airline_codes.add(n)

            error_messages = []

            if bad_flight_numbers:
                error_messages.append(u'Неверные номера рейсов: ' + u'; '.join(bad_flight_numbers))

            if bad_airline_codes:
                error_messages.append(u'Неверные IATA-коды авиакомпаний: ' + u'; '.join(bad_airline_codes))

            if error_messages:
                raise forms.ValidationError(
                    ', '.join(error_messages)
                )

            self.prepared_flight_numbers = prepared_flight_numbers

        return u'\n'.join([n[1] for n in prepared_flight_numbers])

    raw_flight_numbers = forms.CharField(
        label=u'Номера рейсов',
        widget=forms.Textarea(
            attrs={
                'rows': 3
            }
        ),
        required=False,
        help_text=_(u'Номера рейсов можно писать по одному в строке или через запятую')
    )

    class Meta:
        model = FlightReview
        exclude = []


class RelatedAirlineFlightReviewListFilter(admin.SimpleListFilter):
    title = 'Airline'
    parameter_name = 'airline_filter'

    def lookups(self, request, model_admin):
        qs = FlightReview.objects.distinct().order_by('airline__title')
        return qs.values_list('airline_id', 'airline__title')

    def queryset(self, request, queryset):
        if self.value() is None:
            return queryset
        return queryset.filter(airline_id=self.value())


class FlightReviewAdmin(admin.ModelAdmin):
    form = FlightReviewForm

    def get_form(self, request, obj=None, *args, **kwargs):
        form = super(FlightReviewAdmin, self).get_form(request, *args, **kwargs)
        form.base_fields['raw_flight_numbers'].initial = None

        if obj and obj.flight_numbers:
            flight_numbers = '\n'.join([f.flight_number for f in obj.flight_numbers.all()])
            form.base_fields['raw_flight_numbers'].initial = flight_numbers

        return form

    def formfield_for_dbfield(self, db_field, **kwargs):
        if db_field.name in ('flight_numbers', ):
            db = kwargs.get('using')
            kwargs.pop('request', None)
            kwargs['widget'] = VerboseManyToManyRawIdWidget(
                db_field.rel, self.admin_site, using=db
            )

            return db_field.formfield(**kwargs)

        return super(FlightReviewAdmin, self).formfield_for_dbfield(db_field, **kwargs)

    def save_model(self, request, obj, form, change):
        prepared_flight_numbers = getattr(form, 'prepared_flight_numbers', None)

        if prepared_flight_numbers:
            obj.flight_numbers.all().delete()

            for airline, n in prepared_flight_numbers:
                fn, created = FlightNumber.objects.get_or_create(
                    flight_number=n,
                    defaults={
                        'airline': airline,
                    }
                )

                obj.flight_numbers.add(fn)

        obj.moderator = request.user
        obj.moderated_lmt = datetime.now()
        obj.save()

    def make_published(modeladmin, request, queryset):
        queryset.update(enable_show=True)

    def make_unpublished(modeladmin, request, queryset):
        queryset.update(enable_show=False)

    def airline_title(modeladmin, queryset):
        return queryset.airline.title

    def colored_dirty_lang_class(self, obj):
        if obj.dirty_lang_class:
            return '<div style="color:darkred;">%s</div>' % obj.dirty_lang_class.upper()
        return str(obj.id)

    colored_dirty_lang_class.allow_tags = True
    colored_dirty_lang_class.short_description = u'Ругань'

    make_published.short_description = u"Опубликовать"
    make_unpublished.short_description = u"Снять с публикации"

    airline_title.short_description = u'Перевозчик'
    airline_title.admin_order_field = 'airline__title'

    actions = [make_published, make_unpublished]
    list_display = ('colored_dirty_lang_class', 'review_content', 'source', 'airline_title', 'enable_show', )
    raw_id_fields = ('settlement_from', 'settlement_to', 'station_from', 'station_to', 'airline')
    readonly_fields = ('moderator', 'moderated_lmt', "review_id")

    search_fields = (
        'review_content', 'author_name', 'author_email',
    )

    list_filter = (
        'enable_show',
        'source__code',
        'dirty_lang_class',
        RelatedAirlineFlightReviewListFilter,
    )

    fieldset_review = ('Отзыв', {'fields': ("airline", "review_content", "raw_flight_numbers"), })
    fieldset_attrs = ('Атрибуты', {'fields': (
        "review_id", "review_datetime", "review_url", "rating", "author_name",
        "author_email", "source", "settlement_from", "settlement_to",
        "station_from", "station_to", "flight_date",
    )})
    fieldset_moder = ('Модератор', {'fields': ("enable_show", "moderator", "moderated_lmt")})

    inlines = (FlightNumbersInline,)

    fieldsets = (
        fieldset_review,
        fieldset_attrs,
        fieldset_moder,
    )


admin.site.register(FlightReview, FlightReviewAdmin)


class AvgPriceAdmin(RaspExportModelAdmin):
    list_display = ('departure_settlement', 'arrival_settlement',
                    'price', 'currency', 'month_forward', 'year_forward',
                    'passengers')
    raw_id_fields = ('arrival_settlement', 'departure_settlement')
    search_fields = ('departure_settlement__title',)
    list_filter = ('national_version', 'month_forward', 'year_forward',
                   'direct_flight', 'passengers')

admin.site.register(AvgPrice, AvgPriceAdmin)


class AviaWizardDirectionAdmin(RaspExportModelAdmin):
    list_display = ('settlement_1', 'settlement_2')
    raw_id_fields = ('settlement_1', 'settlement_2')
    search_fields = (
        'settlement_1__title', 'settlement_1__title_en', 'settlement_1__title_tr',
        'settlement_2__title', 'settlement_2__title_en', 'settlement_2__title_tr'
    )

admin.site.register(AviaWizardDirection, AviaWizardDirectionAdmin)


class FlightRatingAdmin(RaspExportModelAdmin):
    list_display = (
        'number', 'scores', 'good_count', 'bad_count', 'bad_percent', 'avg_scores',
        'outrunning', 'delayed_less_30', 'delayed_30_60', 'delayed_60_90',
        'delayed_more_90', 'canceled'
    )
    readonly_fields = tuple(list(list_display) + ['scores_description'])
    search_fields = ('number', )

admin.site.register(FlightRating, FlightRatingAdmin)


class CompanyRatingAdmin(RaspExportModelAdmin):
    list_display = (
        'company', 'flight_count', 'scores', 'good_count', 'bad_count',
        'bad_percent', 'avg_scores', 'outrunning', 'delayed_less_30',
        'delayed_30_60', 'delayed_60_90', 'delayed_more_90', 'canceled'
    )
    readonly_fields = list_display
    search_fields = ('company__title', )

admin.site.register(CompanyRating, CompanyRatingAdmin)


class TourDepartureAdmin(RaspExportModelAdmin):
    list_display = ('geo_id', 'settlement')
    raw_id_fields = ('settlement',)
    search_fields = ('geo_id', 'settlement__title')

admin.site.register(TourDeparture, TourDepartureAdmin)


class TourDestinationAdmin(RaspExportModelAdmin):
    list_display = ('geo_id', 'settlement')
    raw_id_fields = ('settlement',)
    search_fields = ('geo_id', 'settlement__title')

admin.site.register(TourDestination, TourDestinationAdmin)


class NearCountriesAdmin(RaspExportModelAdmin):
    list_display = ('country',)
    raw_id_fields = ('neighbours',)

    def formfield_for_dbfield(self, db_field, **kwargs):
        if db_field.name in ('neighbours',):
            db = kwargs.get('using')
            kwargs.pop('request', None)
            kwargs['widget'] = VerboseManyToManyRawIdWidget(
                db_field.rel, self.admin_site, using=db
            )

            return db_field.formfield(**kwargs)

        return super(NearCountriesAdmin, self).formfield_for_dbfield(db_field, **kwargs)

admin.site.register(NearCountries, NearCountriesAdmin)


class SimilarDirectionAdmin(RaspExportModelAdmin):
    list_display = ('direction', 'similar_from', 'similar_to', 'count')
    raw_id_fields = ('similar_from', 'similar_to')
    search_fields = ('direction', )

admin.site.register(SimilarDirection, SimilarDirectionAdmin)


class BestOffersAdmin(RaspExportModelAdmin):
    list_display = ('direction', 'national_version', 'price')
    list_filter = ('national_version',)
    search_fields = ('direction', )

admin.site.register(BestOffers, BestOffersAdmin)


class SeoDirectionAdmin(RaspExportModelAdmin):
    list_display = ('point_from', 'point_to', 'publish')
    raw_id_fields = ('point_from', 'point_to')

admin.site.register(SeoDirection, SeoDirectionAdmin)


class CustomCacheTimeDirectionForm(forms.ModelForm):
    class Meta:
        model = CustomCacheTimeDirection
        exclude = []

    def clean(self):
        cleaned_data = self.cleaned_data

        if not cleaned_data.get('point_from') and not cleaned_data.get('point_to'):
            raise forms.ValidationError(_(u'Хотя бы одна из точек должна быть заполнена'))

        return cleaned_data


class CustomCacheTimeDirectionAdmin(RaspExportModelAdmin):
    list_display = ('point_from', 'point_to', 'roundtrip', 'time', 'enabled')
    raw_id_fields = ('point_from', 'point_to')
    list_filter = ('enabled',)

    form = CustomCacheTimeDirectionForm

admin.site.register(CustomCacheTimeDirection, CustomCacheTimeDirectionAdmin)


class TopDirectionAdmin(RaspExportModelAdmin):
    list_display = ('departure_settlement', 'arrival_settlement', 'count', 'national_version')
    raw_id_fields = ('arrival_settlement', 'departure_settlement')
    search_fields = ('departure_settlement__title',)
    list_filter = ('national_version',)
    ordering = ('national_version', '-count')

admin.site.register(TopDirection, TopDirectionAdmin)


class BalanceRedirectAdmin(RaspExportModelAdmin):
    list_display = ('eventdate', 'eventtime', 'partner', 'national_version', 'pp', 'filtered')
    ordering = ('-eventdate', '-eventtime')
    list_filter = ('partner', 'national_version', 'pp', 'filtered')

admin.site.register(BalanceRedirect, BalanceRedirectAdmin)


class BalanceRedirectPrepearedAdmin(RaspExportModelAdmin):
    list_display = ('eventdate', 'billing_order_id', 'national_version', 'pp', 'count')
    ordering = ('-eventdate', 'billing_order_id', 'national_version', 'pp')
    list_filter = ('billing_order_id', 'national_version', 'pp')

admin.site.register(BalanceRedirectPrepeared, BalanceRedirectPrepearedAdmin)


class ShowLogAdmin(RaspExportModelAdmin):
    list_display = ('eventdate', 'partner', 'national_version', 'pp', 'show_count')
    ordering = ('-eventdate', 'partner', 'national_version', 'pp')
    list_filter = ('partner', 'national_version', 'pp')

admin.site.register(ShowLog, ShowLogAdmin)


class MinPriceAdmin(RaspExportModelAdmin):
    list_display = ('departure_settlement', 'arrival_settlement',
                    'price', 'currency', 'date_forward', 'national_version')
    raw_id_fields = ('arrival_settlement', 'departure_settlement')
    search_fields = ('departure_settlement__title',)
    list_filter = ('national_version',)
    readonly_fields = ('eventtime',)

admin.site.register(MinPrice, MinPriceAdmin)


class MinCountryPriceAdmin(RaspExportModelAdmin):
    list_display = ('departure_country', 'arrival_country',
                    'price', 'currency', 'date_forward', 'national_version')
    raw_id_fields = ('departure_country', 'arrival_country')
    search_fields = ('departure_country__title',)
    list_filter = ('national_version',)
    readonly_fields = ('eventtime',)

admin.site.register(MinCountryPrice, MinCountryPriceAdmin)


class ReadOnlyFieldsForm(forms.ModelForm):
    readonly_fields = ()

    def __init__(self, *args, **kwargs):
        super(ReadOnlyFieldsForm, self).__init__(*args, **kwargs)
        for field in (field for name, field in self.fields.iteritems() if name in self.readonly_fields):
            field.widget.attrs['disabled'] = 'disabled'
            field.required = False

    def clean(self):
        cleaned_data = super(ReadOnlyFieldsForm, self).clean()
        for field in self.readonly_fields:
            cleaned_data[field] = getattr(self.instance, field)

        return cleaned_data


class PartnerReviewAdminForm(ReadOnlyFieldsForm):
    redirect_params = forms.CharField(widget=widgets.AdminTextareaWidget())
    mbo_data = forms.CharField(widget=widgets.AdminTextareaWidget())

    user_info = forms.CharField(
        widget=PreTagWidget(attrs={'style': 'float: left;'})
    )

    readonly_fields = (
        'redirect_params',
        'user_info',
        'mbo_data',
    )

    class Meta:
        model = PartnerReview
        exclude = []


class PartnerReviewAdmin(RaspExportModelAdmin):
    form = PartnerReviewAdminForm

    def get_columns(self):
        return [
            'id',
            'result',
            'description',
            'partner__code',
            'hit_time',
            'review_time',
            'currency__code',
            'price',
            'price_mbo',
            'price_diff_abs',
            'price_diff_rel',
            'settlement_from_id',
            'station_from_id',
            'settlement_to_id',
            'station_to_id',
            'date_forward',
            'date_backward',
            'adults',
            'children',
            'infants',
            'order_content',
        ]

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)
        urlpatterns = super(PartnerReviewAdmin, self).get_urls()
        info = self.model._meta.app_label, self.model._meta.model_name
        urlpatterns = [
            url(r'^stat/$', wrap(self.stat_view), name='%s_%s_stat' % info),
        ] + urlpatterns

        return urlpatterns

    def stat_view(self, request):
        model = self.model
        opts = model._meta
        app_label = opts.app_label
        media = self.media

        now = request.now.replace(second=0, microsecond=0, tzinfo=None)

        def add_month(dt, months):
            return dt + relativedelta(months=months)

        periods = (
            ('h', 'Час', now - timedelta(hours=1)),
            ('3h', '3 часа', now - timedelta(hours=3)),
            ('12h', '12 часов', now - timedelta(hours=12)),
            ('24h', '24 часа', now - timedelta(hours=24)),
            ('3d', '3 дня', now - timedelta(days=3)),
            ('w', 'Неделя', now - timedelta(weeks=1)),
            ('m', 'Месяц', add_month(now, -1)),
            ('3m', '3 месяца', add_month(now, -3)),
            ('6m', '6 месяцев', add_month(now, -6)),
            ('y', 'Год', now.replace(year=now.year-1)),
        )

        p_codes, p_names, p_boundaries = zip(*periods)

        periods_boundaries = dict(zip(p_codes, p_boundaries))

        period = request.GET.get('period')

        if period not in p_codes:
            period = '12h'

        periods_links = [
            ('?period=%s' % code if code != period else None, name)
            for code, name in zip(p_codes, p_names)
        ]

        period_boundary = periods_boundaries[period]

        reviews = model.objects.filter(review_time__gt=period_boundary)

        partners_ids = reviews.values_list('partner_id', flat=True).distinct()

        partners = sorted(Partner.objects.filter(id__in=partners_ids),
                          key=lambda p: p.L_title())

        reviews_counts = dict(
            reviews.
            values_list('partner_id').annotate(Count('id')).order_by()
        )

        warnings_counts = dict(
            reviews.filter(result='pass').exclude(price_diff_abs=0).
            values_list('partner_id').annotate(Count('id')).order_by()
        )

        problems_counts = dict(
            reviews.filter(result='problem').
            values_list('partner_id').annotate(Count('id')).order_by()
        )

        errors_counts = dict(
            reviews.filter(result='error').
            values_list('partner_id').annotate(Count('id')).order_by()
        )

        def partnerreview_link(p, **kwargs):
            kwargs['partner__id__exact'] = p.id

            return '/admin/order/partnerreview/?' + urlencode(kwargs)

        def json_stat(partner):
            p = partner
            reviews = reviews_counts.get(p.id, 0)

            return {
                'partner': p,
                'reviews_count': reviews,
                'warnings_count': int(100 * warnings_counts.get(p.id, 0) / reviews),
                'problems_count': int(100 * problems_counts.get(p.id, 0) / reviews),
                'errors_count': int(100 * errors_counts.get(p.id, 0) / reviews),
                'reviews_count_link': partnerreview_link(
                    p, review_time__gt=period_boundary,
                ),
                'warnings_count_link': partnerreview_link(
                    p, review_time__gt=period_boundary,
                    result='pass', description='Цена почти совпадает'
                ),
                'problems_count_link': partnerreview_link(
                    p, review_time__gt=period_boundary,
                    result='problem'
                ),
                'errors_count_link': partnerreview_link(
                    p, review_time__gt=period_boundary,
                    result='error'
                ),
            }

        context = {
            'title': _(u'Статистика проверок цен от партнёров'),
            'media': media,
            'app_label': opts.app_label,
            'opts': opts,
            'has_absolute_url': hasattr(self.model, 'get_absolute_url'),
            'content_type_id': ContentType.objects.get_for_model(self.model).id,
            'stat': map(json_stat, partners),
            'periods_links': periods_links,
        }

        return TemplateResponse(
            request,
            "admin/%s/%s/stat.html" % (app_label, opts.object_name.lower()),
            context,
            current_app=self.admin_site.name
        )

    def partner_link(self, obj):
        return u'<a href="../partner/%s">%s</a>' % (
            obj.partner.id,
            obj.partner.code
        )

    partner_link.allow_tags = True
    partner_link.short_description = _(u'Партнёр')

    def result_flag(self, obj):
        _ = obj.result == 'pass'  # noqa

        result_icons = {
            'pass': 'icon-yes.gif',
            'problem': 'icon-no.gif',
            'error': 'icon-unknown.gif',
        }

        return u'<img src="/static/admin/img/%s" alt="%s"> %s' % (
            result_icons.get(obj.result, 'icon_alert.gif'), obj.result, obj.get_result_display()
        )

    result_flag.allow_tags = True
    result_flag.short_description = _(u'Результат проверки')

    def point_from(self, obj):
        return obj.station_from or obj.settlement_from

    point_from.short_description = _(u'Пункт вылета')

    def point_to(self, obj):
        return obj.station_to or obj.settlement_to

    point_to.short_description = _(u'Пункт прилёта')

    fieldsets = (
        (u'', {'fields': [
            (
                'result',
                'description',
            ),
            'partner',
        ]}),
        (_(u'Цена'), {'fields': [
            (
                'price',
                'currency'
            ),
            'price_mbo',
            'price_diff_abs',
            'price_diff_rel',
        ]}),
        (_(u'Показ'), {'fields': [
            'order_content',
            ('settlement_from', 'station_from'),
            ('settlement_to', 'station_to'),
            (
                'date_forward',
                'date_backward',
            ),
            'adults',
            'children',
            'infants',
        ]}),
        (_(u'Проверка'), {'fields': [
            'hit_time',
            'review_time',
            'mbo_data',
            'redirect_params',
            'user_info',
        ]}),
    )

    list_display = (
        'review_time',
        'partner_link',
        'result_flag',
        'description',
        'price',
        'price_diff_abs',
        'price_diff_rel',
        'point_from',
        'point_to',
    )

    list_filter = (CustomPartnerReviewResultFilter, CustomPartnerReviewPartnersFilter)

    raw_id_fields = (
        'partner', 'currency',
        'settlement_from', 'station_from',
        'settlement_to', 'station_to',
    )

    readonly_fields = (
        'result',
        'description',
        'partner',
        'hit_time',
        'review_time',
        'price',
        'price_mbo',
        'price_diff_abs',
        'price_diff_rel',
        'currency',
        'settlement_from',
        'station_from',
        'settlement_to',
        'station_to',
        'order_content',
        'date_forward',
        'date_backward',
        'adults',
        'children',
        'infants',
    )


admin.site.register(PartnerReview, PartnerReviewAdmin)


class TopFlightAdmin(admin.ModelAdmin):
    list_display = (
        'from_point_key', 'to_point_key', 'day_of_week',
        'national_version', 'redirects', 'flights',
    )
    search_fields = ('from_point_key', )
    list_filter = ('national_version', 'day_of_week')

admin.site.register(TopFlight, TopFlightAdmin)


class ToursMatrixAdmin(admin.ModelAdmin):
    list_display = ('from_geo_id', 'to_geo_id')

admin.site.register(ToursMatrix, ToursMatrixAdmin)


class CurrencyLangAdmin(admin.ModelAdmin):
    list_display = ('title', 'code', 'enable')
    ordering = ('title', )

admin.site.register(CurrencyLang, CurrencyLangAdmin)


class CurrencyTranslationAdmin(admin.ModelAdmin):
    list_display = ('currency', 'lang', 'title')
    ordering = ('currency', )

admin.site.register(CurrencyTranslation, CurrencyTranslationAdmin)


class CurrencyTranslationInline(admin.TabularInline):
    model = CurrencyTranslation


class CurrencyAdmin(admin.ModelAdmin):
    inlines = (
        CurrencyTranslationInline,
    )
    list_display = ('title', 'code', 'iso_code', 'enable')
    ordering = ('title', )

admin.site.register(Currency, CurrencyAdmin)


class CompanyTariffInline(admin.TabularInline):
    """ Чтобы в компании отображать тарифы """
    model = CompanyTariff
    extra = 0


class AviaCompanyAdmin(RaspExportModelAdmin):
    fields = [
        'rasp_company',
        'iata',
        'cost_type',
        'baggage_rules',
        'baggage_rules_are_valid',
        'baggage_rules_url',
        (
            'baggage_length',
            'baggage_width',
            'baggage_height',
        ),
        'baggage_dimensions_sum',
        (
            'carryon_length',
            'carryon_width',
            'carryon_height',
        ),
        'carryon_dimensions_sum',
        ('baggage_min_price', 'baggage_max_price', 'baggage_price_currency'),
        'baggage_description_url',
    ]
    raw_id_fields = ('rasp_company',)
    inlines = (CompanyTariffInline,)

    def response_change(self, request, obj, *args, **kwargs):
        result = super(AviaCompanyAdmin, self).response_change(request, obj, *args, **kwargs)
        self.after_saving_inlines_hook(request, obj)
        return result

    def response_add(self, request, obj, post_url_continue=None):
        result = super(AviaCompanyAdmin, self).response_add(request, obj, post_url_continue)
        self.after_saving_inlines_hook(request, obj)
        return result

    def after_saving_inlines_hook(self, request, obj):
        obj.ensure_has_default_tariff()

admin.site.register(AviaCompany, AviaCompanyAdmin)


class CompanyTariffAdmin(RaspExportModelAdmin):
    list_display = (
        '__unicode__', 'published', 'mask',
        'baggage_allowed', 'baggage_norm', 'baggage_pieces',
        'carryon', 'carryon_norm',
    )
    list_filter = ('published', 'avia_company',)

    fields = [
        'avia_company',
        'mask', 'description',
        ('baggage_allowed', 'baggage_norm', 'baggage_pieces'),
        ('carryon', 'carryon_norm'),
        'published',
    ]

    def publish(self, request, queryset):
        rows_updated = queryset.update(published=True)
        self.message_user(
            request, _(u'Тарифы опубликованы: {count}').
            format(count=rows_updated)
        )
    publish.short_description = _(u'Опубликоавать')

    def unpublish(self, request, queryset):
        rows_updated = queryset.update(published=False)
        self.message_user(
            request, _(u'Тарифы скрыты: {count}').format(count=rows_updated)
        )
    unpublish.short_description = _(u'Скрыть')

    actions = ['publish', 'unpublish']

admin.site.register(CompanyTariff, CompanyTariffAdmin)


class AviaCompanyModelForm(forms.ModelForm):
    def has_changed(self):
        # Создавать авиакомпанию даже без правок
        return True


class AviaCompanyInline(admin.StackedInline):
    model = AviaCompany
    form = AviaCompanyModelForm
    fields = ('cost_type',)
    extra = 0
    show_change_link = True


class RegistryAdmin(admin.ModelAdmin):
    list_display = ('key', 'val')


admin.site.register(Registry, RegistryAdmin)


class RazladkiLastProcessedTableAdmin(admin.ModelAdmin):
    list_display = ('log_name', 'timestamp', 'offset', 'mode', 'yt_path')


admin.site.register(RazladkiLastProcessedTable, RazladkiLastProcessedTableAdmin)


# ============ Selfbook rules ============


class SelfBookNationalVersionInline(admin.TabularInline):
    model = SelfBookNationalVersion
    extra = 0
    show_change_link = False


class SelfBookPartnerInline(admin.TabularInline):
    model = SelfBookPartner
    raw_id_fields = ('partner',)
    extra = 0
    show_change_link = False


class SelfBookCompanyInline(admin.TabularInline):
    model = SelfBookCompany
    raw_id_fields = ('company',)
    extra = 1
    show_change_link = False


class SelfBookDirectionInline(admin.TabularInline):
    model = SelfBookDirection
    raw_id_fields = (
        'station_from',
        'settlement_from',
        'country_from',
        'station_to',
        'settlement_to',
        'country_to',
    )
    extra = 1
    show_change_link = True


class SelfBookRuleAdmin(admin.ModelAdmin):
    inlines = (
        SelfBookNationalVersionInline,
        SelfBookPartnerInline,
        SelfBookCompanyInline,
        SelfBookDirectionInline,
    )

admin.site.register(SelfBookRule, SelfBookRuleAdmin)

admin.site.register(NationalVersion)
