# -*- encoding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import logging
import math

from travel.avia.library.python.avia_data.models import NearDirection, AviaDirection
from travel.avia.library.python.common.utils import geo
from travel.avia.library.python.common.models.geo import Settlement, CodeSystem
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log

log = logging.getLogger(__name__)


# Эти константы должны быть синхронизированы
SEARCH_DISTANCE = 350  # км
SEARCH_DISTANCE_DEGREE = 4  # 4 градуса это ~400км, т.е. берём на всякий случай с запасом


def get_settlements_within(settlements, center_point, step, distance):
    min_lat = center_point.latitude - step
    max_lat = center_point.latitude + step
    min_lng = center_point.longitude - step
    max_lng = center_point.longitude + step

    settlements = [s for s in settlements if (
        s != center_point
        and min_lat <= s.latitude <= max_lat
        and min_lng <= s.longitude <= max_lng
    )]

    for s in settlements:
        s.distance = geo.great_circle_distance_km(center_point, s)

    settlements.sort(key=lambda _s: _s.distance)

    return [s for s in settlements if s.distance <= distance]


def _main():
    log.info('precache')
    CodeSystem.objects.precache()

    all_settlements = Settlement.objects.filter(
        hidden=False,
        type_choices__contains='plane',
        latitude__isnull=False,
        longitude__isnull=False
    )

    all_settlements = filter(lambda x: x.iata_point or x.sirena_point, all_settlements)
    all_settlements_dict = {s.id: s for s in all_settlements}

    all_directions = AviaDirection.objects.filter(
        departure_settlement__latitude__isnull=False,
        departure_settlement__longitude__isnull=False,
        arrival_settlement__latitude__isnull=False,
        arrival_settlement__longitude__isnull=False
    )

    settlement_pairs = {
        '%d_%d' % (d[0], d[1]): True
        for d in all_directions.values_list('departure_settlement', 'arrival_settlement')
    }

    arrival_settlements = list(all_directions.order_by('arrival_settlement').values_list(
        'arrival_settlement', flat=True
    ).distinct())

    log.info('process %s directions and %s arrival settlements',
             all_directions.count(), len(arrival_settlements))

    near_bulk = []
    count = 0

    for arrival_id in arrival_settlements:
        if count % 50 == 0:
            log.info('processed %s arrival cities', count)

        count += 1

        arrival_settlement = all_settlements_dict.get(arrival_id)
        if not arrival_settlement:
            continue

        settlement_within = get_settlements_within(
            all_settlements, arrival_settlement,
            step=SEARCH_DISTANCE_DEGREE, distance=SEARCH_DISTANCE
        )

        if not len(settlement_within):
            continue

        departure_settlements_ids = list(AviaDirection.objects.filter(
            arrival_settlement_id=arrival_id,
            departure_settlement__latitude__isnull=False,
            departure_settlement__longitude__isnull=False
        ).order_by('departure_settlement').values_list(
            'departure_settlement', flat=True
        ).distinct())

        for departure_id in departure_settlements_ids:
            tmp_within = [
                i for i in settlement_within
                if settlement_pairs.get('%d_%d' % (departure_id, i.id))
            ]

            if not len(tmp_within):
                continue

            min_distance = math.ceil(tmp_within[0].distance / 50.0) * 50
            max_distance = math.ceil(tmp_within[-1].distance / 50.0) * 50
            default_distance = min_distance
            # это ближайшее значение от 50 до 350 км с шагом в 50 км,
            # в котором есть не более трех городов
            step = min_distance
            while step < (max_distance + 50):
                stations_count = len([x for x in tmp_within if x.distance])
                if stations_count > 2:
                    break

                default_distance = step
                step += 50

            near_bulk.append(NearDirection(
                departure_settlement_id=departure_id,
                arrival_settlement_id=arrival_id,
                min_distance=min_distance,
                default_distance=default_distance,
                max_distance=max_distance
            ))

    log.info('delete all')
    NearDirection.objects.all().delete()

    log.info('Save %s directions', len(near_bulk))
    NearDirection.objects.bulk_create(near_bulk, batch_size=1000)


def main():
    from optparse import OptionParser

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout(log)

    create_current_file_run_log()

    _main()
