# -*- encoding: utf-8 -*-
import functools
import sys

from copy import copy
from datetime import datetime, date
from optparse import Option, OptionParser, OptionValueError

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger


def check_number(option, opt, value):
    try:
        days = int(value)

    except ValueError:
        raise OptionValueError(
            'option %s: bad number: %r' % (opt, value))

    return days


# Convert station to settlement
def fix_point(point_key, stations_map):
    if point_key.startswith('s'):
        return stations_map.get(point_key)

    return point_key


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.db.models import Sum
    from django.db import transaction

    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper, last_logs_tables

    from travel.avia.library.python.avia_data.models import AviaDirection, AviaSettlement
    from travel.avia.library.python.common.models.geo import Settlement, Station
    from travel.avia.library.python.route_search.models import ZNodeRoute2
    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def min_price_map(stations_map, plane_settlements, record):
        if not record.get('direction'):
            return
        try:
            from_key, to_key = record['direction'].split('_')
        except ValueError:
            return

        from_key = fix_point(from_key, stations_map)
        to_key = fix_point(to_key, stations_map)

        if 'stops' in record:
            forward_direct_flight = int(record['stops'].startswith('0'))
            bacward_direct_flight = int(record['stops'].endswith('_0'))

        else:
            forward_direct_flight = None
            bacward_direct_flight = None

        plane_points = from_key in plane_settlements and to_key in plane_settlements

        if plane_points:
            yield {
                'direction': '%s_%s' % (from_key, to_key),
                'direct_flight': forward_direct_flight,
                'iso_eventtime': record['iso_eventtime']
            }

            if record.get('date_backward'):
                yield {
                    'direction': '%s_%s' % (to_key, from_key),
                    'direct_flight': bacward_direct_flight,
                    'iso_eventtime': record['iso_eventtime']
                }

    def min_price_reduce(search_days, key, records):
        today = date.today()

        direct_flight_count = 0
        connecting_flight_count = 0
        popularity_count = 0

        for r in records:
            direct_flight = r['direct_flight']

            # Считаем популярность в зависимости от древности
            popularity = 10
            iso_eventtime = datetime.strptime(r['iso_eventtime'], '%Y-%m-%d %H:%M:%S').date()
            delta = (today - iso_eventtime).days
            popularity_count += popularity * (search_days - delta) / search_days + 1

            if direct_flight == 'None':
                continue

            if bool(int(direct_flight)):
                direct_flight_count += 1

            else:
                connecting_flight_count += 1

        yield {
            'direction': key['direction'],
            'direct_flights': direct_flight_count,
            'connecting_flights': connecting_flight_count,
            'popularity': popularity_count,
        }

    def precached_objects():
        stations = Station.objects.filter(
            t_type__code='plane',
            settlement_id__isnull=False,
            hidden=False
        )

        stations = [s for s in stations if s.iata or s.sirena_id]

        station_map = {s.point_key: s.settlement.point_key for s in stations}

        plane_settlements = {
            s.settlement.point_key for s in stations
        }.union(
            s.settlement.id for s in stations
        )

        return station_map, plane_settlements

    def build_traffic(yt, stations_map, plane_settlements, tmp_table, log, search_days):
        source_tables = last_logs_tables(yt, '//home/rasp/logs/rasp-min-price-log', options.days)

        log.info('Process tables: %s -> %s', source_tables, tmp_table)
        yt.run_map_reduce(
            mapper=functools.partial(min_price_map, stations_map, plane_settlements),
            reducer=functools.partial(min_price_reduce, search_days),
            source_table=source_tables,
            destination_table=tmp_table,
            reduce_by=['direction'],
        )

    @transaction.atomic
    def store_traffic(tmp_table, log):
        log.info('Store yt results to DB')
        AviaDirection.objects.all().delete()

        for record in yt.read_table(tmp_table, format=yt.DsvFormat(), raw=False):
            try:
                from_id, to_id = record['direction'].split('_')
                direct_flights = int(record['direct_flights'])
                connecting_flights = int(record['connecting_flights'])
                popularity = int(record['popularity'])

                avia_direction = AviaDirection(
                    departure_settlement_id=int(from_id[1:]),
                    arrival_settlement_id=int(to_id[1:]),
                    direct_flights=direct_flights,
                    connecting_flights=connecting_flights,
                    popularity=popularity
                )

                avia_direction.save()

            except Exception:
                log.exception('Error while save avia direction')
                log.error('%r', record)
                continue

    @transaction.atomic
    def add_rasp_routes(plane_settlements, log):
        log.info('Read ZNodeRoute2')
        routes = ZNodeRoute2.objects.filter(
            route__t_type__code='plane',
        ).exclude(
            settlement_from__isnull=True
        ).exclude(
            settlement_to__isnull=True,
        ).values_list(
            'settlement_from',
            'settlement_to'
        ).distinct()

        avia_directions = AviaDirection.objects.all().values_list(
            'departure_settlement', 'arrival_settlement'
        )

        log.info('Prepare bulk data from %s grouped ZNodeRoutes', len(routes))
        avia_bulk = []

        for x, (departure_settlement_id, arrival_settlement_id) in enumerate(routes):
            plane_pair = all([departure_settlement_id in plane_settlements,
                              arrival_settlement_id in plane_settlements])
            new_pair = (departure_settlement_id, arrival_settlement_id) not in avia_directions

            if all([plane_pair, new_pair]):
                avia_bulk.append(
                    AviaDirection(
                        departure_settlement_id=departure_settlement_id,
                        arrival_settlement_id=arrival_settlement_id,
                    )
                )

            if x % 10000 == 0:
                log.info('%s / %s records: %s new routes', x, len(routes), len(avia_bulk))

        log.info('Bulk insert %s directions', len(avia_bulk))
        AviaDirection.objects.bulk_create(avia_bulk, batch_size=1000)

    def _get_settlements_popularity(direction):
        settlements_ids = AviaDirection.objects.filter(
            popularity__gt=0
        ).order_by(
            direction + '_settlement'
        ).values_list(
            direction + '_settlement', flat=True
        ).distinct()

        return Settlement.objects.filter(
            id__in=settlements_ids
        ).annotate(
            popularity=Sum('air_traffic_' + direction + '_settlement__popularity')
        ).values('id', 'popularity')

    @transaction.atomic
    def store_settlements_popularity():
        log.info('Store settlements popularity to DB')
        settlements_bulk = []

        AviaSettlement.objects.all().delete()

        popularities = _get_settlements_popularity('departure')

        for item in popularities:
            settlements_bulk.append(AviaSettlement(
                settlement_id=item['id'], arrival=False, popularity=item['popularity']
            ))

        popularities = _get_settlements_popularity('arrival')

        for item in popularities:
            settlements_bulk.append(AviaSettlement(
                settlement_id=item['id'], arrival=True, popularity=item['popularity']
            ))

        AviaSettlement.objects.bulk_create(settlements_bulk, batch_size=1000)

    # BEGIN main()
    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-d', '--days', dest='days', type='days', help='number of last logs to aggregate', default=365)

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)

        stations_map, plane_settlements = precached_objects()

        log.info('Number of days: %s' % options.days)
        log.info('%s airports with settlemet loaded' % len(stations_map))
        log.info('%s plane settlemets loaded' % len(plane_settlements))

        log.info('---------------------------------')

        tmp_table = yt.create_temp_table()

        build_traffic(yt, stations_map, plane_settlements, tmp_table, log, options.days)
        store_traffic(tmp_table, log)

        add_rasp_routes(plane_settlements, log)

        yt.remove(tmp_table)

        store_settlements_popularity()

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')


class Yoption(Option):
    TYPES = Option.TYPES + ('days', 'quantity', 'countries')
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER['days'] = check_number
    TYPE_CHECKER['quantity'] = check_number
