# -*- coding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import logging

from datetime import datetime, timedelta

from django.db import transaction
from django.conf import settings

from travel.avia.library.python.avia_data.models import BestOffers, MinPrice
from travel.avia.admin.avia_scripts.models.www import Route, RThread  # noqa
from travel.avia.library.python.common.utils.date import RunMask
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log

log = logging.getLogger(__name__)


def load_min_prices():
    min_price_dict = {}

    min_prices = MinPrice.objects.filter(
        date_forward__gte=datetime.now(),
        date_forward__lte=datetime.now() + timedelta(days=30),
        passengers='1_0_0',
    ).order_by(
        'price'
    )

    for mp in min_prices:
        mp_key = (mp.departure_settlement_id, mp.arrival_settlement_id, mp.national_version)

        if mp_key not in min_price_dict:
            min_price_dict[mp_key] = []

        min_price_dict[mp_key].append(mp)

    return min_price_dict


def load_year_days():
    year_days_dict = {}

    for t in RThread.objects.values("number", "year_days"):
        route_number = t["number"]
        year_days = t["year_days"]

        if route_number not in year_days_dict:
            year_days_dict[route_number] = []

        if route_number and t["year_days"]:
            year_days_dict[route_number].append(year_days)

    return year_days_dict


def check_route(route, dt, year_days_dict):
    if not route:
        return False

    year_days_set = set()

    for yd in year_days_dict.get(route, []):
        year_days_set.add(yd)

    if not any([RunMask.runs_at(yd, dt) for yd in year_days_set]):
        return False

    return True


def check_routes(routes, dt, year_days_dict):
    if not dt:
        return False

    for route in routes:
        if not check_route(route, dt, year_days_dict):
            return False

    return True


def process_direction(direction, min_price_dict, rthreads_dict, quantity):
    mp_key = (
        direction['departure_settlement'],
        direction['arrival_settlement'],
        direction['national_version'],
    )

    min_prices = min_price_dict.get(mp_key, [])

    best_offers_bulk = []

    for mp in min_prices:
        if len(best_offers_bulk) >= quantity:
            break

        all_routes = mp.routes.split('/')

        if len(all_routes) == 2:
            forward_routes, backward_routes = all_routes
        else:
            forward_routes, backward_routes = all_routes[0], None

        splitted_forward_routes = forward_routes.split(';')
        splitted_backward_routes = backward_routes.split(';') if backward_routes else None

        if backward_routes and len(splitted_forward_routes) != len(splitted_backward_routes):
            continue

        for x in range(0, len(splitted_forward_routes)):
            if len(best_offers_bulk) >= quantity:
                break

            forward_route = splitted_forward_routes[x]

            if splitted_backward_routes:
                backward_route = splitted_backward_routes[x]
            else:
                backward_route = None

            if not forward_route:
                continue

            if not check_routes(forward_route.split(','), mp.date_forward, rthreads_dict):
                continue

            if backward_route and not check_routes(backward_route.split(','), mp.date_backward, rthreads_dict):
                continue

            bo = BestOffers(
                direction='c%s_c%s' % (mp.departure_settlement_id, mp.arrival_settlement_id),
                forward_date=mp.date_forward,
                backward_date=mp.date_backward,
                forward_route=forward_route,
                backward_route=backward_route,
                price=mp.price,
                currency=mp.currency,
                national_version=mp.national_version,
            )

            best_offers_bulk.append(bo)

    BestOffers.objects.bulk_create(best_offers_bulk, batch_size=1000)


@transaction.atomic
def _main(options):
    log.info('Start')

    log.info('Preload year_days')
    year_days_dict = load_year_days()
    log.info('Preload min prices')
    min_price_dict = load_min_prices()

    BestOffers.objects.all().delete()

    for national_version in settings.AVIA_NATIONAL_VERSIONS:
        log.info('Process national_version: %s', national_version)

        directions = MinPrice.objects.filter(
            national_version=national_version
        ).values('departure_settlement', 'arrival_settlement', 'national_version').distinct()

        log.info('Build %s directions' % len(directions))

        for x, direction in enumerate(directions):
            if options.verbose and x % 1000 == 0:
                log.info('Direction %s / %s' % (x, len(directions)))

            process_direction(direction, min_price_dict, year_days_dict, options.quantity)

        count = BestOffers.objects.filter(national_version=national_version).count()
        log.info('%s offers saved for %s' % (count, national_version))

    count = BestOffers.objects.all().count()
    log.info('%s offers saved total' % count)
    log.info('Done')


def main():
    from optparse import OptionParser

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-q', '--quantity', type='int', default=5)
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    create_current_file_run_log()

    _main(options)
