# -*- encoding: utf-8 -*-
import sys

from copy import copy
from datetime import datetime
from optparse import Option, OptionParser, OptionValueError

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger


def check_number(option, opt, value):
    try:
        days=int(value)

    except ValueError:
        raise OptionValueError(
            "option %s: bad number: %r" % (opt, value))

    return days


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.db import transaction

    from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables
    from travel.avia.admin.stats.models import DynamicCache

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def map_table(record):
        iso_eventtime = datetime.strptime(record['iso_eventtime'], '%Y-%m-%d %H:%M:%S')
        prices_requested = int(record['prices_requested'])
        prices_found = int(record['prices_found'])

        if prices_found:
            hit_percent = int(float(prices_found) / prices_requested * 100)

        else:
            hit_percent = 0

        yield {
            'national_version': record['national_version'],
            'eventdate': iso_eventtime.date().strftime('%Y-%m-%d'),
            'percent': hit_percent,
            'month_number': record['iso_eventtime'][:7],
        }

    def reduce_table(key, records):
        records_count = 0
        percent_sum = 0

        for r in records:
            records_count += 1
            percent_sum += int(r['percent'])

        yield {
            'national_version': key['national_version'],
            'eventdate': key['eventdate'],
            'percent': int(float(percent_sum) / records_count),
            'count': records_count,
            'month_number': key['month_number']
        }

    def build_flexible_cache(tmp_table, log):
        logs_quantity = options.days
        source_tables = yt_last_logs_tables(
            '//home/rasp/logs/rasp-dynamic-cache-log',
            logs_quantity
        )

        log.info('Map tables: %s -> %s', source_tables, tmp_table)
        yt.run_map_reduce(
            mapper=map_table,
            reducer=reduce_table,
            source_table=source_tables,
            destination_table=tmp_table,
            reduce_by=['national_version', 'eventdate', 'month_number'],
        )

    @transaction.atomic
    def store_min_prices(tmp_table, log):
        log.info('Read and store flexible cache from %s to DB', tmp_table)
        DynamicCache.objects.all().delete()

        for record in yt.read_table(tmp_table, format=yt.JsonFormat(), raw=False):
            national_version = record['national_version']
            eventdate = datetime.strptime(record['eventdate'], '%Y-%m-%d')
            percent = int(record['percent'])
            count = int(record['count'])

            flexible_cache = DynamicCache(
                national_version=national_version,
                eventdate=eventdate,
                month_year=record['month_number'],
                percent=percent,
                count=count
            )

            flexible_cache.save()

    # BEGIN main()
    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-d", "--days", dest="days", type="days", help="number of last logs to aggregate", default=14)

    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)

        log.info('Number of days: %s' % options.days)
        log.info('---------------------------------')

        tmp_table_flexible = yt.create_temp_table()

        build_flexible_cache(tmp_table_flexible, log)
        store_min_prices(tmp_table_flexible, log)

        yt.remove(tmp_table_flexible)

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')


class Yoption(Option):
    TYPES = Option.TYPES + ("days", "quantity", "countries")
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER["days"] = check_number
    TYPE_CHECKER["quantity"] = check_number
