# -*- encoding: utf-8 -*-
import functools
import json

from copy import copy
from collections import defaultdict
from datetime import datetime, timedelta
from optparse import Option, OptionParser, OptionValueError

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

MIN_COUNT = 7  # Минимальное число рейсов за 30 дней
DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S'
KEY_COLUMNS = ['airline_code', 'flight_number']


def check_number(option, opt, value):
    try:
        days = int(value)

    except ValueError:
        raise OptionValueError(
            "option %s: bad number: %r" % (opt, value))

    return days


def _select_fields(dct, fields):
    return {
        field: dct[field]
        for field in fields
    }


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.db import transaction

    from travel.avia.library.python.avia_data.models import FlightRating
    from travel.avia.library.python.common.models.schedule import Company
    from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def map_tablo(left_date_time, right_date_time, rasp_id_to_iata, rasp_id_to_sirena, record):
        airline_code = record['_rest'].get('airline_code')
        airline_id = record['airline_id']
        if (
            record['direction'] != 'arrival' or
            record['result'] != 'success' or
            not record['time_scheduled'] or
            (record['status'] != 'cancelled' and not record['time_actual']) or
            (not airline_id and not airline_code)
        ):
            return

        if not airline_code:
            airline_code = rasp_id_to_iata.get(airline_id)
        if not airline_code:
            airline_code = rasp_id_to_sirena.get(airline_id)
        if not airline_code:
            return

        try:
            # Проверяем формат
            arrival = datetime.strptime(record['time_scheduled'], DATETIME_FORMAT)
        except ValueError:
            return

        # Проверим диапазон дат
        if not (left_date_time <= arrival <= right_date_time):
            return

        result = _select_fields(
            record,
            ['time_actual', 'time_scheduled', 'unixtime', 'status', 'flight_number'],
        )
        result.update({
            'airline_code': airline_code,
        })

        yield result

    def max_by_unixtime(key, records):
        yield max(records, key=lambda x: x['unixtime'])

    def red_score(key, records):
        results = defaultdict(int)
        total_score = 0
        log_lines = []

        for r in records:
            scheduled_arrival_str = r['time_scheduled']
            actual_arrival_str = r['time_actual']
            arrival_cancelled = r['status'] == 'cancelled'

            if arrival_cancelled:
                result_key = 'cancelled'
                score = 0

            else:
                scheduled_arrival = datetime.strptime(scheduled_arrival_str, DATETIME_FORMAT)
                actual_arrival = datetime.strptime(actual_arrival_str, DATETIME_FORMAT)

                arrival_delta = float((actual_arrival - scheduled_arrival).total_seconds()) / 3600

                # задержка прилета на 3 и более часов (4 балла)
                if arrival_delta >= 3:
                    result_key = 'delayed_more_90'
                    score = 2

                # задержка 1-3 часа (3 балла)
                elif arrival_delta >= 1:
                    result_key = 'delayed_60_90'
                    score = 4

                # задержка 30 мин -1 час (2 балла)
                elif arrival_delta >= 0.5:
                    result_key = 'delayed_30_60'
                    score = 6

                # задержка прилета на 15-29 (1 балл)
                elif arrival_delta >= 0.25:
                    result_key = 'delayed_less_30'
                    score = 8

                # прилеты на более полчаса раньше считаем без баллов
                elif arrival_delta < 0:
                    result_key = 'outrunning'
                    score = 10

                # иначе - все хорошо
                else:
                    result_key = 'good'
                    score = 10

            results[result_key] += 1
            results['total'] += 1

            total_score += score

            log_lines.append(
                '%s %s planned: %s, real: %s, rule: %s, score: %s' % (
                    key['airline_code'],
                    key['flight_number'],
                    scheduled_arrival_str,
                    actual_arrival_str,
                    result_key,
                    score,
                )
            )

        yield {
            'airline_code': key['airline_code'],
            'flight_number': key['flight_number'],
            'results': json.dumps(dict(results)),
            'log': json.dumps(sorted(log_lines)),
            'total_score': total_score
        }

    @transaction.atomic
    def save_results(tmp_table_results):
        log.info('Write results')

        FlightRating.objects.all().delete()

        to_insert = []
        for record in yt.read_table(tmp_table_results, format=yt.YsonFormat()):
            results = json.loads(record['results'])
            log_lines = json.loads(record['log'])
            scores = record['total_score']

            description = '%s\n\n%s' % (
                json.dumps(results),
                '\n'.join(log_lines)
            )

            total = results.get('total', 0)
            good = results.get('good', 0) + results.get('outrunning', 0)
            bad = total - good

            if total > MIN_COUNT:
                number = u'{} {}'.format(
                    record['airline_code'].decode('utf-8'),
                    record['flight_number'].decode('utf-8'),
                )
                avg_scores = float(scores) / total

                flight_rating = FlightRating(
                    number=number,
                    scores=scores,
                    good_count=good,
                    bad_count=bad,
                    bad_percent=float(bad) / total * 100,
                    avg_scores=avg_scores,
                    scores_description=description,
                    outrunning=float(results.get('outrunning', 0)) / total * 100,
                    delayed_less_30=float(results.get('delayed_less_30', 0)) / total * 100,
                    delayed_30_60=float(results.get('delayed_30_60', 0)) / total * 100,
                    delayed_60_90=float(results.get('delayed_60_90', 0)) / total * 100,
                    delayed_more_90=float(results.get('delayed_more_90', 0)) / total * 100,
                    canceled=float(results.get('cancelled', 0)) / total * 100
                )

                to_insert.append(flight_rating)

        if to_insert:
            FlightRating.objects.bulk_create(to_insert)

    # BEGIN main()
    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-d', '--days', dest='days', type='days', help='number of last logs to aggregate', default=30)

    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    configure_wrapper(yt)

    log.info('Number of days: %d', options.days)
    log.info('---------------------------------')

    source_tables = yt_last_logs_tables('//home/logfeller/logs/avia-update-flight-status-log/1d', options.days)
    tmp_table = yt.create_temp_table()
    tmp_table_results = yt.create_temp_table()

    # Что бы исключить головные боли с конвертацией таймзон
    # возьмем интевал options.days с небольшой погрешностью
    left_date_time = datetime.now() - timedelta(days=options.days + 1)
    right_date_time = datetime.now() - timedelta(days=1)

    log.info('Run tablo map')
    rasp_id_to_iata = dict(Company.objects.values_list('id', 'iata'))
    rasp_id_to_sirena = dict(Company.objects.values_list('id', 'sirena_id'))
    yt.run_map_reduce(
        mapper=functools.partial(map_tablo, left_date_time, right_date_time, rasp_id_to_iata, rasp_id_to_sirena),
        reducer=max_by_unixtime,
        source_table=source_tables,
        destination_table=tmp_table,
        reduce_by=KEY_COLUMNS + ['time_scheduled'],
    )

    log.info('Run score reduce')
    yt.run_map_reduce(
        mapper=None,
        reducer=red_score,
        source_table=tmp_table,
        destination_table=tmp_table_results,
        reduce_by=KEY_COLUMNS,
    )

    save_results(tmp_table_results)

    yt.remove(tmp_table)
    yt.remove(tmp_table_results)

    log.info('Done')


class Yoption(Option):
    TYPES = Option.TYPES + ("days", "quantity", "countries")
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER["days"] = check_number
    TYPE_CHECKER["quantity"] = check_number
